--------------------------
-- ALIASES ---------------
--------------------------
-- MATH ALIASES ----------
local min, max = math.min, math.max
-- LUA ALIASES -----------
local error = error
--------------------------
-- UTILS -----------------
--------------------------
-- CHECK MODNAME PREFIX --
---@source https://github.com/luanti-org/luanti/blob/master/builtin/game/register.lua#L58
local function check_modname_prefix(name)
	if name:sub(1,1) == ":" then
		-- If the name starts with a colon, we can skip the modname prefix
		-- mechanism.
		return name:sub(2)
	else
		-- Enforce that the name starts with the correct mod name.
		local expected_prefix = core.get_current_modname() .. ":"
		if name:sub(1, #expected_prefix) ~= expected_prefix then
			error("Name " .. name .. " does not follow naming conventions: " ..
				"\"" .. expected_prefix .. "\" or \":\" prefix required")
		end

		-- Enforce that the name only contains letters, numbers and underscores.
		local subname = name:sub(#expected_prefix+1)
		if subname:find("[^%w_]") then
			error("Name " .. name .. " does not follow naming conventions: " ..
				"contains unallowed characters")
		end

		return name
	end
end
--------------------------
-- REGISTRIES ------------
--------------------------
-- SPAWN REGISTRY --------
natural_entities.registered_spawns = {}
---@param name string The registered spawn's name
---@param def table A spawn group definition table
--- spawn group definition table from `example.txt`:
---    spawn_rate = 1.0,
---    -- on average how often these entities will spawn.
---    -- in units of hz, # of spawns per second.
---
---    min_y = 0,
---    -- minimum y entities can spawn at.
---
---    max_y = 0,
---    -- maximum y entities can spawn at.
---
---    entities = {
---        ["modname:entity"] = 1.0
---    },
---    -- entities that this spawn group spawns.
---    -- format is ["modname:entity"] = chance_to_spawn.
---
---    check = function (position, ent_name) 
---        return true
---    end
---    -- custom check function.
---    -- only stops entities from spawning if the function returns false, nil does not stop entities from spawning.
function natural_entities.register_spawn_group(name, def)
	-- validate name
	if name == nil then
		error("Unable to register spawn: Name is nil")
	end
	name = check_modname_prefix(name)

	if def.spawn_rate <= 0.0 then error("While registering spawn group " .. name .. ": invalid spawn rate") end
	if def.min_y > def.max_y then
		core.log("warning", "while registering spawn group " .. name .. ": min_y is greater than max_y")
	end

	def.min_y = min(def.min_y or 0, def.max_y or 0)
	def.max_y = max(def.min_y or 0, def.max_y or 0)
	def.check = def.check or function() end

	if type(def.check) ~= "function" then error("While registering spawn group " .. name .. ": check must be a function or nil") end
	if not def.entities then error("While registering spawn group " .. name .. ": entities table is missing") end
	if type(def.entities) ~= "table" then error("While registering spawn group " .. name .. ": entities must be a table") end
	
	natural_entities.registered_spawns[name] = def
end
--- Alias for backwards compatibility.
natural_entities.register_spawn = natural_entities.register_spawn_group
--------------------------