-- Parameters

local SINGLENODE = true -- Use singlenode mapgen and spawnplayer function
local YMIN = -33000 -- Approximate base of realm stone
local YMAX = 33000 -- Approximate top of atmosphere / mountains / floatlands
local TERCEN = -128 -- Terrain zero level, average seabed
local YWAT = 1 -- Sea surface y
local YSAV = 5 -- Average sandline y, dune grasses above this
local SAMP = 3 -- Sandline amplitude
local YCLOMIN = 207 -- Minimum height of mod clouds
local CLOUDS = true -- Mod clouds?

local TERSCA = 512 -- Vertical terrain scale
local XLSAMP = 0.1 -- Extra large scale height variation amplitude
local BASAMP = 0.3 -- Base terrain amplitude
local MIDAMP = 0.1 -- Mid terrain amplitude
local CANAMP = 0.4 -- Canyon terrain maximum amplitude
local ATANAMP = 1.1 -- Arctan function amplitude,
					-- smaller = more and larger floatlands above ridges
local BLENEXP = 2 -- Terrain blend exponent

local TSTONE = 0.02 -- Density threshold for stone, depth of soil at TERCEN
local TRIVER = -0.028 -- Densitybase threshold for river surface
local TRSAND = -0.035 -- Densitybase threshold for river sand
local TSTREAM = -0.004 -- Densitymid threshold for stream surface
local TSSAND = -0.005 -- Densitymid threshold for stream sand
local TLAVA = 2 -- Maximum densitybase threshold for lava,
				-- small because grad is non-linear
local TFIS = 0.01 -- Fissure threshold, controls width
local TSEAM = 0.2 -- Seam threshold, width of seams
local ORESCA = 512 -- Seam system vertical scale
local ORETHI = 0.002 -- Ore seam thickness tuner
local BERGDEP = 32 -- Maximum iceberg depth
local TFOG = -0.04 -- Fog top densitymid threshold

local biomeparams = {
	HITET = 0.35, -- High temperature threshold
	LOTET = -0.35, -- Low ..
	ICETET = -0.7, -- Ice ..
	HIHUT = 0.35, -- High humidity threshold
	LOHUT = -0.35, -- Low ..
	FOGHUT = 1.0, -- Fog ..
	BLEND = 0.02, -- Biome blend randomness
}

local flora = {
	PINCHA = 36, -- Pine tree 1/x chance per node
	APTCHA = 36, -- Appletree
	FLOCHA = 289, -- Flower
	GRACHA = 36, -- Grassland grasses
	JUTCHA = 16, -- Jungletree
	JUGCHA = 16, -- Junglegrass
	CACCHA = 2209, -- Cactus
	DRYCHA = 121, -- Dry shrub
	ACACHA = 1369, -- Acacia tree
	GOGCHA = 9, -- Golden grass
	PAPCHA = 4, -- Papyrus
	DUGCHA = 16, -- Dune grass
}

-- 3D noises

-- 3D noise for terrain

local np_terrain = {
	offset = 0,
	scale = 1,
	spread = {x = 384, y = 192, z = 384},
	seed = 593,
	octaves = 5,
	persist = 0.67
}

-- 3D noise for fissures

local np_fissure = {
	offset = 0,
	scale = 1,
	spread = {x = 256, y = 512, z = 256},
	seed = 20099,
	octaves = 5,
	persist = 0.5
}

-- 3D noise for temperature

local np_temp = {
	offset = 0,
	scale = 1,
	spread = {x = 1024, y = 1024, z = 1024},
	seed = 9130,
	octaves = 3,
	persist = 0.5
}

-- 3D noise for humidity

local np_humid = {
	offset = 0,
	scale = 1,
	spread = {x = 1024, y = 1024, z = 1024},
	seed = -55500,
	octaves = 3,
	persist = 0.5
}

-- 3D noise for ore seam networks

local np_seam = {
	offset = 0,
	scale = 1,
	spread = {x = 512, y = 512, z = 512},
	seed = -992221,
	octaves = 2,
	persist = 0.5
}

-- 3D noise for rock strata inclination

local np_strata = {
	offset = 0,
	scale = 1,
	spread = {x = 512, y = 512, z = 512},
	seed = 92219,
	octaves = 3,
	persist = 0.5
}

-- 2D noises

-- 2D noise for mid terrain / streambed height

local np_mid = {
	offset = 0,
	scale = 1,
	spread = {x = 768, y = 768, z = 768},
	seed = 85546,
	octaves = 5,
	persist = 0.5
}

-- 2D noise for base terrain / riverbed height

local np_base = {
	offset = 0,
	scale = 1,
	spread = {x = 4096, y = 4096, z = 4096},
	seed = 8890,
	octaves = 3,
	persist = 0.33
}

-- 2D noise for extra large scale height variation

local np_xlscale = {
	offset = 0,
	scale = 1,
	spread = {x = 8192, y = 8192, z = 8192},
	seed = -72,
	octaves = 3,
	persist = 0.33
}

-- 2D noise for magma surface

local np_magma = {
	offset = 0,
	scale = 1,
	spread = {x = 128, y = 128, z = 128},
	seed = -13,
	octaves = 2,
	persist = 0.5
}


-- Do files

dofile(minetest.get_modpath("watershed") .. "/nodes.lua")
dofile(minetest.get_modpath("watershed") .. "/functions.lua")

-- Initialize 3D and 2D noise objects to nil

local nobj_terrain = nil
local nobj_fissure = nil
local nobj_temp    = nil
local nobj_humid   = nil
local nobj_seam    = nil
local nobj_strata  = nil
	
local nobj_mid     = nil
local nobj_base    = nil
local nobj_xlscale = nil
local nobj_magma   = nil


-- Localise noise buffers

local nbuf_terrain = {}
local nbuf_fissure = {}
local nbuf_temp = {}
local nbuf_humid = {}
local nbuf_seam = {}
local nbuf_strata = {}
	
local nbuf_mid = {}
local nbuf_base = {}
local nbuf_xlscale = {}
local nbuf_magma = {}


-- Localise data buffer

local dbuf = {}


-- Mapchunk generation function

function watershed_chunkgen(x0, y0, z0, x1, y1, z1, area, data)
	local c_air = minetest.get_content_id("air")
	local c_ignore = minetest.get_content_id("ignore")
	local c_water = minetest.get_content_id("default:water_source")
	local c_sand = minetest.get_content_id("default:sand")
	local c_desand = minetest.get_content_id("default:desert_sand")
	local c_snowblock = minetest.get_content_id("default:snowblock")
	local c_ice = minetest.get_content_id("default:ice")
	local c_dirtsnow = minetest.get_content_id("default:dirt_with_snow")
	local c_jungrass = minetest.get_content_id("default:junglegrass")
	local c_dryshrub = minetest.get_content_id("default:dry_shrub")
	local c_danwhi = minetest.get_content_id("flowers:dandelion_white")
	local c_danyel = minetest.get_content_id("flowers:dandelion_yellow")
	local c_rose = minetest.get_content_id("flowers:rose")
	local c_tulip = minetest.get_content_id("flowers:tulip")
	local c_geranium = minetest.get_content_id("flowers:geranium")
	local c_viola = minetest.get_content_id("flowers:viola")
	local c_stodiam = minetest.get_content_id("default:stone_with_diamond")
	local c_mese = minetest.get_content_id("default:mese")
	local c_stogold = minetest.get_content_id("default:stone_with_gold")
	local c_stocopp = minetest.get_content_id("default:stone_with_copper")
	local c_stoiron = minetest.get_content_id("default:stone_with_iron")
	local c_stocoal = minetest.get_content_id("default:stone_with_coal")
	local c_sandstone = minetest.get_content_id("default:sandstone")
	local c_gravel = minetest.get_content_id("default:gravel")
	local c_clay = minetest.get_content_id("default:clay")
	local c_grass5 = minetest.get_content_id("default:grass_5")
	local c_obsidian = minetest.get_content_id("default:obsidian")
	
	local c_wsfreshwater = minetest.get_content_id("watershed:freshwater")
	local c_wsmixwater = minetest.get_content_id("watershed:mixwater")
	local c_wsstone = minetest.get_content_id("watershed:stone")
	local c_wsredstone = minetest.get_content_id("watershed:redstone")
	local c_wsgrass = minetest.get_content_id("watershed:grass")
	local c_wsdrygrass = minetest.get_content_id("watershed:drygrass")
	local c_wsgoldengrass = minetest.get_content_id("watershed:goldengrass")
	local c_wsdirt = minetest.get_content_id("watershed:dirt")
	local c_wspermafrost = minetest.get_content_id("watershed:permafrost")
	local c_wslava = minetest.get_content_id("watershed:lava")
	local c_wsfreshice = minetest.get_content_id("watershed:freshice")
	local c_wscloud = minetest.get_content_id("watershed:cloud")
	local c_wsluxore = minetest.get_content_id("watershed:luxore")
	local c_wsicydirt = minetest.get_content_id("watershed:icydirt")

	-- perlinmap stuff
	local sidelen = x1 - x0 + 1
	local chulensxyz = {x = sidelen, y = sidelen + 2, z = sidelen}
	local chulensxz = {x = sidelen, y = sidelen, z = 1}
	local minposxyz = {x = x0, y = y0 - 1, z = z0}
	local minposxz = {x = x0, y = z0}

	-- 3D and 2D noise objects created once on first mapchunk generation only
	nobj_terrain = nobj_terrain or minetest.get_perlin_map(np_terrain, chulensxyz)
	nobj_fissure = nobj_fissure or minetest.get_perlin_map(np_fissure, chulensxyz)
	nobj_temp    = nobj_temp    or minetest.get_perlin_map(np_temp, chulensxyz)
	nobj_humid   = nobj_humid   or minetest.get_perlin_map(np_humid, chulensxyz)
	nobj_seam    = nobj_seam    or minetest.get_perlin_map(np_seam, chulensxyz)
	nobj_strata  = nobj_strata  or minetest.get_perlin_map(np_strata, chulensxyz)
	
	nobj_mid     = nobj_mid     or minetest.get_perlin_map(np_mid, chulensxz)
	nobj_base    = nobj_base    or minetest.get_perlin_map(np_base, chulensxz)
	nobj_xlscale = nobj_xlscale or minetest.get_perlin_map(np_xlscale, chulensxz)
	nobj_magma   = nobj_magma   or minetest.get_perlin_map(np_magma, chulensxz)

	-- 3D and 2D perlinmaps created per mapchunk
	local nvals_terrain = nobj_terrain:get3dMap_flat(minposxyz, nbuf_terrain)
	local nvals_fissure = nobj_fissure:get3dMap_flat(minposxyz, nbuf_fissure)
	local nvals_temp    = nobj_temp   :get3dMap_flat(minposxyz, nbuf_temp)
	local nvals_humid   = nobj_humid  :get3dMap_flat(minposxyz, nbuf_humid)
	local nvals_seam    = nobj_seam   :get3dMap_flat(minposxyz, nbuf_seam)
	local nvals_strata  = nobj_strata :get3dMap_flat(minposxyz, nbuf_strata)
	
	local nvals_mid     = nobj_mid    :get2dMap_flat(minposxz, nbuf_mid)
	local nvals_base    = nobj_base   :get2dMap_flat(minposxz, nbuf_base)
	local nvals_xlscale = nobj_xlscale:get2dMap_flat(minposxz, nbuf_xlscale)
	local nvals_magma   = nobj_magma  :get2dMap_flat(minposxz, nbuf_magma)

	-- ungenerated chunk below?
	local viu = area:index(x0, y0 - 1, z0)
	local ungen = data[viu] == c_ignore

	-- mapgen loop
	local nixyz = 1 -- 3D and 2D perlinmap indexes
	local nixz = 1
	local stable = {} -- stability table of true/false.
					-- is node supported from below by 2 stone or nodes on 2 stone?
	local under = {} -- biome table.
					-- biome number of previous fine material placed in column
	for z = z0, z1 do
		for y = y0 - 1, y1 + 1 do
			local vi = area:index(x0, y, z)
			local viu = area:index(x0, y - 1, z)
			for x = x0, x1 do
				local si = x - x0 + 1 -- stable, under tables index
				-- noise values for node
				local n_absterrain = math.abs(nvals_terrain[nixyz])
				local n_fissure = nvals_fissure[nixyz]
				local n_temp = nvals_temp[nixyz]
				local n_humid = nvals_humid[nixyz]
				local n_seam = nvals_seam[nixyz]
				local n_strata = nvals_strata[nixyz]
				
				local n_absmid = math.abs(nvals_mid[nixz])
				local n_absbase = math.abs(nvals_base[nixz])
				local n_xlscale = nvals_xlscale[nixz]
				local n_magma = nvals_magma[nixz]
				-- get densities
				local n_invbase = (1 - n_absbase)
				local terblen = (math.max(n_invbase, 0)) ^ BLENEXP
				local grad = math.atan((TERCEN - y) / TERSCA) * ATANAMP
				local densitybase = n_invbase * BASAMP + n_xlscale * XLSAMP + grad
				local densitymid = n_absmid * MIDAMP + densitybase
				local canexp = 0.5 + terblen * 0.5
				local canamp = terblen * CANAMP
				local density = n_absterrain ^ canexp * canamp * n_absmid + densitymid
				-- other values
				local triver = TRIVER * n_absbase -- river threshold
				local trsand = TRSAND * n_absbase -- river sand
				local tstream = TSTREAM * (1 - n_absmid) -- stream threshold
				local tssand = TSSAND * (1 - n_absmid) -- stream sand
				local tstone = TSTONE * (1 + grad) -- stone threshold
				local tlava = TLAVA * (1 - n_magma ^ 4 * terblen ^ 16 * 0.6) -- lava threshold
				local ysand = YSAV + n_fissure * SAMP + math.random() * 2 -- sandline
				local bergdep = math.abs(n_seam) * BERGDEP -- iceberg depth
				
				local nofis = false -- set fissure bool
				if math.abs(n_fissure) >= TFIS then
					nofis = true
				end
				
				local biome = false -- select biome for node
				if n_temp < biomeparams.LOTET +
						(math.random() - 0.5) * biomeparams.BLEND then
					if n_humid < biomeparams.LOHUT +
							(math.random() - 0.5) * biomeparams.BLEND then
						biome = 1 -- tundra
					elseif n_humid > biomeparams.HIHUT +
							(math.random() - 0.5) * biomeparams.BLEND then
						biome = 3 -- taiga
					else
						biome = 2 -- snowy plains
					end
				elseif n_temp > biomeparams.HITET +
						(math.random() - 0.5) * biomeparams.BLEND then
					if n_humid < biomeparams.LOHUT +
							(math.random() - 0.5) * biomeparams.BLEND then
						biome = 7 -- desert
					elseif n_humid > biomeparams.HIHUT +
							(math.random() - 0.5) * biomeparams.BLEND then
						biome = 9 -- rainforest
					else
						biome = 8 -- savanna
					end
				else
					if n_humid < biomeparams.LOHUT then
						biome = 4 -- dry grassland
					elseif n_humid > biomeparams.HIHUT then
						biome = 6 -- deciduous forest
					else
						biome = 5 -- grassland
					end
				end
				
				-- overgeneration and in-chunk generation
				if y == y0 - 1 then -- node layer below chunk, initialise tables
					under[si] = 0
					if ungen then
						if nofis and density >= 0 then -- if node solid
							stable[si] = 2
						else
							stable[si] = 0
						end
					else -- scan top layer of chunk below
						local nodid = data[vi]
						if nodid == c_wsstone
								or nodid == c_wsredstone
								or nodid == c_wsdirt
								or nodid == c_wspermafrost
								or nodid == c_wsluxore
								or nodid == c_sand
								or nodid == c_desand
								or nodid == c_mese
								or nodid == c_stodiam
								or nodid == c_stogold
								or nodid == c_stocopp
								or nodid == c_stoiron
								or nodid == c_stocoal
								or nodid == c_sandstone
								or nodid == c_gravel
								or nodid == c_clay
								or nodid == c_obsidian then
							stable[si] = 2
						else
							stable[si] = 0
						end
					end
				elseif y >= y0 and y <= y1 then -- chunk
					-- add nodes and flora
					if densitybase >= tlava then -- lava
						if densitybase >= 0 then
							data[vi] = c_wslava
						end
						stable[si] = 0
						under[si] = 0
					elseif densitybase >= tlava - math.min(0.6 + density * 6, 0.6)
							and density < tstone then -- obsidian
						data[vi] = c_obsidian
						stable[si] = 1
						under[si] = 0
					elseif density >= tstone and nofis  -- stone cut by fissures
							or (density >= tstone and density < TSTONE * 1.2 and
							y <= YWAT) -- stone around water
							or (density >= tstone and density < TSTONE * 1.2 and
							densitybase >= triver ) -- stone around river
							or (density >= tstone and density < TSTONE * 1.2 and
							densitymid >= tstream ) then -- stone around stream
						local densitystr = n_strata * 0.25 + (TERCEN - y) / ORESCA
						-- periodic strata 'density'
						local densityper = densitystr - math.floor(densitystr)
						if (densityper >= 0.05 and densityper <= 0.09) -- sandstone strata
								or (densityper >= 0.25 and densityper <= 0.28)
								or (densityper >= 0.45 and densityper <= 0.47)
								or (densityper >= 0.74 and densityper <= 0.76)
								or (densityper >= 0.77 and densityper <= 0.79)
								or (densityper >= 0.84 and densityper <= 0.87)
								or (densityper >= 0.95 and densityper <= 0.98) then
							data[vi] = c_sandstone
						elseif biome == 7 and density < TSTONE * 3 then
							-- desert stone as surface layer
							data[vi] = c_wsredstone
						elseif math.abs(n_seam) < TSEAM then
							-- ore seams
							if densityper >= 0 and
									densityper <= ORETHI * 4 then
								data[vi] = c_stocoal
							elseif densityper >= 0.3 and
									densityper <= 0.3 + ORETHI * 4 then
								data[vi] = c_stocoal
							elseif densityper >= 0.5 and
									densityper <= 0.5 + ORETHI * 4 then
								data[vi] = c_stocoal
							elseif densityper >= 0.8 and
									densityper <= 0.8 + ORETHI * 4 then
								data[vi] = c_stocoal
							elseif densityper >= 0.55 and
									densityper <= 0.55 + ORETHI * 2 then
								data[vi] = c_gravel
							elseif densityper >= 0.1 and
									densityper <= 0.1 + ORETHI * 2 then
								data[vi] = c_wsluxore
							elseif densityper >= 0.2 and
									densityper <= 0.2 + ORETHI * 2 and
									math.random(2) == 2 then
								data[vi] = c_stoiron
							elseif densityper >= 0.65 and
									densityper <= 0.65 + ORETHI * 2 and
									math.random(2) == 2 then
								data[vi] = c_stoiron
							elseif densityper >= 0.4 and
									densityper <= 0.4 + ORETHI * 2 and
									math.random(3) == 2 then
								data[vi] = c_stocopp
							elseif densityper >= 0.6 and
									densityper <= 0.6 + ORETHI and
									math.random(5) == 2 then
								data[vi] = c_stogold
							elseif densityper >= 0.7 and
									densityper <= 0.7 + ORETHI and
									math.random(7) == 2 then
								data[vi] = c_mese
							elseif densityper >= 0.9 and
									densityper <= 0.9 + ORETHI and
									math.random(11) == 2 then
								data[vi] = c_stodiam
							else
								data[vi] = c_wsstone
							end
						else
							data[vi] = c_wsstone
						end
						stable[si] = stable[si] + 1
						under[si] = 0
					-- fine materials
					elseif density >= 0 and density < tstone and stable[si] >= 2 then
						if y == YWAT - 2 and math.abs(n_temp) < 0.05 then -- clay
							data[vi] = c_clay
						elseif y <= ysand then -- seabed/beach/dune sand not cut by fissures
							data[vi] = c_sand
							under[si] = 10 -- beach/dunes
						elseif densitybase >= trsand + math.random() * 0.001 then -- river sand
							data[vi] = c_sand
							under[si] = 11 -- riverbank papyrus
						elseif densitymid >= tssand + math.random() * 0.001 then -- stream sand
							data[vi] = c_sand
							under[si] = 0
						elseif nofis then -- fine materials cut by fissures
							if biome == 1 then
								data[vi] = c_wspermafrost
								under[si] = 1 -- tundra
							elseif biome == 2 then
								data[vi] = c_wsdirt
								under[si] = 2 -- snowy plains
							elseif biome == 3 then
								data[vi] = c_wsdirt
								under[si] = 3 -- taiga
							elseif biome == 4 then
								data[vi] = c_wsdirt
								under[si] = 4 -- dry grassland
							elseif biome == 5 then
								data[vi] = c_wsdirt
								under[si] = 5 -- grassland
							elseif biome == 6 then
								data[vi] = c_wsdirt
								under[si] = 6 -- forest
							elseif biome == 7 then
								data[vi] = c_desand
								under[si] = 7 -- desert
							elseif biome == 8 then
								data[vi] = c_wsdirt
								under[si] = 8 -- savanna
							elseif biome == 9 then
								data[vi] = c_wsdirt
								under[si] = 9 -- rainforest
							end
						else -- fissure
							stable[si] = 0
							under[si] = 0
						end
					elseif y >= YWAT - bergdep and y <= YWAT + bergdep / 8 -- icesheet
							and n_temp < biomeparams.ICETET and density < tstone
							and math.abs(n_fissure) > 0.01 then
						data[vi] = c_ice
						under[si] = 12
						stable[si] = 0
					elseif y <= YWAT and density < tstone then -- sea water
						data[vi] = c_water
						under[si] = 0
						stable[si] = 0
					-- river water not in fissures
					elseif densitybase >= triver and density < tstone then
						if n_temp < biomeparams.ICETET then
							data[vi] = c_wsfreshice
						else
							if y == YWAT + 1 then
								data[vi] = c_wsmixwater
							else
								data[vi] = c_wsfreshwater
							end
						end
						stable[si] = 0
						under[si] = 0
					-- stream water not in fissures
					elseif densitymid >= tstream and density < tstone then
						if n_temp < biomeparams.ICETET then
							data[vi] = c_wsfreshice
						else
							if y == YWAT + 1 then
								data[vi] = c_wsmixwater
							else
								data[vi] = c_wsfreshwater
							end
						end
						stable[si] = 0
						under[si] = 0
					-- air above surface node
					elseif density < 0 and y >= YWAT and under[si] ~= 0 then
						local fnoise = n_fissure -- noise for flower colours
						if under[si] == 1 then
							data[viu] = c_wsicydirt
							if math.random(flora.DRYCHA) == 2 then
								data[vi] = c_dryshrub
							end
						elseif under[si] == 2 then
							data[viu] = c_dirtsnow
							data[vi] = c_snowblock
						elseif under[si] == 3 then
							if math.random(flora.PINCHA) == 2 then
								watershed_pinetree(x, y, z, area, data)
							else
								data[viu] = c_dirtsnow
								data[vi] = c_snowblock
							end
						elseif under[si] == 4 then
							data[viu] = c_wsdrygrass
							if math.random(flora.DRYCHA) == 2 then
								data[vi] = c_dryshrub
							end
						elseif under[si] == 5 then
							data[viu] = c_wsgrass
							if math.random(flora.FLOCHA) == 2 then
								watershed_flower(data, vi, fnoise)
							elseif math.random(flora.GRACHA) == 2 then
								data[vi] = c_grass5
							end
						elseif under[si] == 6 then
							if math.random(flora.APTCHA) == 2 then
								watershed_appletree(x, y, z, area, data)
							else
								data[viu] = c_wsgrass
								if math.random(flora.FLOCHA) == 2 then
									watershed_flower(data, vi, fnoise)
								elseif math.random(flora.GRACHA) == 2 then
									data[vi] = c_grass5
								end
							end
						elseif under[si] == 7 and n_temp < biomeparams.HITET + 0.1 then
							if math.random(flora.CACCHA) == 2 then
								watershed_cactus(x, y, z, area, data)
							elseif math.random(flora.DRYCHA) == 2 then
								data[vi] = c_dryshrub
							end
						elseif under[si] == 8 then
							if math.random(flora.ACACHA) == 2 then
								watershed_acaciatree(x, y, z, area, data)
							else
								data[viu] = c_wsdrygrass
								if math.random(flora.GOGCHA) == 2 then
									data[vi] = c_wsgoldengrass
								end
							end
						elseif under[si] == 9 then
							if math.random(flora.JUTCHA) == 2 then
								watershed_jungletree(x, y, z, area, data)
							else
								data[viu] = c_wsgrass
								if math.random(flora.JUGCHA) == 2 then
									data[vi] = c_jungrass
								end
							end
						elseif under[si] == 10 then -- dunes
							if math.random(flora.DUGCHA) == 2 and y > YSAV 
									and biome >= 4 then
								data[vi] = c_wsgoldengrass
							end
						elseif under[si] == 11 and n_temp > biomeparams.HITET then -- hot biome riverbank
							if math.random(flora.PAPCHA) == 2 then
								watershed_papyrus(x, y, z, area, data)
							end
						-- snowy iceberg
						elseif under[si] == 12 and n_humid > biomeparams.LOHUT +
								(math.random() - 0.5) * biomeparams.BLEND then
							data[vi] = c_snowblock
						end
						stable[si] = 0
						under[si] = 0
					elseif density < 0 and densitymid > TFOG and
							n_humid > biomeparams.FOGHUT then -- fog
						data[vi] = c_wscloud
						stable[si] = 0
						under[si] = 0
					elseif density < 0 and CLOUDS and y == y1 and y >= YCLOMIN then -- clouds
						local xrq = 16 * math.floor((x - x0) / 16) -- quantise to 16x16 lattice
						local zrq = 16 * math.floor((z - z0) / 16)
						local yrq = 79
						local qixyz = zrq * 6400 + yrq * 80 + xrq + 1 -- quantised 3D index
						if math.abs(nvals_fissure[qixyz]) < nvals_humid[qixyz] * 0.1 then
							data[vi] = c_wscloud
						end
						stable[si] = 0
						under[si] = 0
					else -- air
						stable[si] = 0
						under[si] = 0
					end
				elseif y == y1 + 1 then -- plane of nodes above chunk
					-- if air above fine materials
					if density < 0 and y >= YWAT and under[si] ~= 0 then
						if under[si] == 1 then -- add surface nodes to chunk top layer
							data[viu] = c_wsicydirt
						elseif under[si] == 2 then
							data[viu] = c_dirtsnow
							data[vi] = c_snowblock -- snowblocks in chunk above
						elseif under[si] == 3 then
							data[viu] = c_dirtsnow
							data[vi] = c_snowblock
						elseif under[si] == 4 then
							data[viu] = c_wsdrygrass
						elseif under[si] == 5 then
							data[viu] = c_wsgrass
						elseif under[si] == 6 then
							data[viu] = c_wsgrass
						elseif under[si] == 8 then
							data[viu] = c_wsdrygrass
						elseif under[si] == 9 then
							data[viu] = c_wsgrass
						end
					end
				end
				nixyz = nixyz + 1
				nixz = nixz + 1
				vi = vi + 1
				viu = viu + 1
			end
			nixz = nixz - sidelen
		end
		nixz = nixz + sidelen
	end
end


-- Regenerate chunk by chat command. Dependant on chunksize = 5.

minetest.register_chatcommand("regen",{
	description = "Regenerate player's current mapchunk",
	privs = {server = true, rollback = true},
	func = function(name, params)
		local t1 = os.clock()

		local player = minetest.get_player_by_name(name)
		local pos = player:getpos()
		local plax = math.floor(pos.x + 0.5)
		local play = math.floor(pos.y + 0.5)
		local plaz = math.floor(pos.z + 0.5)
		local x0 = (80 * math.floor((plax + 32) / 80)) - 32
		local y0 = (80 * math.floor((play + 32) / 80)) - 32
		local z0 = (80 * math.floor((plaz + 32) / 80)) - 32
		local x1 = x0 + 79
		local y1 = y0 + 79
		local z1 = z0 + 79

		if y0 < YMIN or y1 > YMAX then
			return
		end

		print ("[watershed] regenerate mapchunk")

		local vm = minetest.get_voxel_manip()
		local pos1 = {x = x0, y = y0 - 1, z = z0}
		local pos2 = {x = x1, y = y1 + 1, z = z1}
		local emin, emax = vm:read_from_map(pos1, pos2)
		local area = VoxelArea:new({MinEdge = emin, MaxEdge = emax})
		local data = vm:get_data(dbuf)

		watershed_chunkgen(x0, y0, z0, x1, y1, z1, area, data)	

		vm:set_data(data)
		vm:write_to_map()
		vm:update_map()

		local chugent = math.ceil((os.clock() - t1) * 1000)
		print ("[watershed] " .. chugent .. " ms")
	end
})


-- On generated function

minetest.register_on_generated(function(minp, maxp, seed)
	if minp.y < YMIN or maxp.y > YMAX then
		return
	end

	local t1 = os.clock()

	local x1 = maxp.x
	local y1 = maxp.y
	local z1 = maxp.z
	local x0 = minp.x
	local y0 = minp.y
	local z0 = minp.z

	local vm, emin, emax = minetest.get_mapgen_object("voxelmanip")
	local area = VoxelArea:new{MinEdge = emin, MaxEdge = emax}
	local data = vm:get_data(dbuf)

	watershed_chunkgen(x0, y0, z0, x1, y1, z1, area, data)

	vm:set_data(data)
	if not SINGLENODE then
		vm:set_lighting({day = 0, night = 0}) -- remove incorrect precalculated light
	end
	vm:calc_lighting()
	vm:write_to_map(data)
	vm:update_liquids()
	
	local chugent = math.ceil((os.clock() - t1) * 1000)
	print ("[watershed] " .. chugent .. " ms")
end)


-- Singlenode option

if SINGLENODE then
	-- Set mapgen parameters

	minetest.set_mapgen_params({mgname = "singlenode", flags = "nolight"})


	-- Spawn player function. Requires chunksize = 80 nodes (the default)

	function spawnplayer(player)
		local xsp
		local ysp
		local zsp

		local nobj_terrain = nil
		local nobj_mid     = nil
		local nobj_base    = nil
		local nobj_xlscale = nil

		local nbuf_terrain
		local nbuf_mid
		local nbuf_base
		local nbuf_xlscale

		for chunk = 1, 128 do
			print ("[watershed] searching for spawn "..chunk)
			local x0 = 80 * math.random(-32, 32) - 32
			local z0 = 80 * math.random(-32, 32) - 32
			local y0 = -32
			local x1 = x0 + 79
			local z1 = z0 + 79
			local y1 = 47
			local sidelen = 80
			local chulensxyz = {x = sidelen, y = sidelen + 2, z = sidelen}
			local chulensxz = {x = sidelen, y = sidelen, z = 1}
			local minposxyz = {x = x0, y = y0 - 1, z = z0}
			local minposxz = {x = x0, y = z0}

			nobj_terrain = nobj_terrain or minetest.get_perlin_map(np_terrain, chulensxyz)
			nobj_mid     = nobj_mid     or minetest.get_perlin_map(np_mid, chulensxz)
			nobj_base    = nobj_base    or minetest.get_perlin_map(np_base, chulensxz)
			nobj_xlscale = nobj_xlscale or minetest.get_perlin_map(np_xlscale, chulensxz)

			local nvals_terrain = nobj_terrain:get3dMap_flat(minposxyz, nbuf_terrain)
			local nvals_mid     = nobj_mid    :get2dMap_flat(minposxz, nbuf_mid)
			local nvals_base    = nobj_base   :get2dMap_flat(minposxz, nbuf_base)
			local nvals_xlscale = nobj_xlscale:get2dMap_flat(minposxz, nbuf_xlscale)

			local nixz = 1
			local nixyz = 1
			for z = z0, z1 do
				for y = y0, y1 do
					for x = x0, x1 do
						local n_absterrain = math.abs(nvals_terrain[nixyz])
						local n_absmid = math.abs(nvals_mid[nixz])
						local n_absbase = math.abs(nvals_base[nixz])
						local n_xlscale = nvals_xlscale[nixz]
						
						local n_invbase = (1 - n_absbase)
						local terblen = (math.max(n_invbase, 0)) ^ BLENEXP
						local grad = math.atan((TERCEN - y) / TERSCA) * ATANAMP
						local densitybase = n_invbase * BASAMP + n_xlscale * XLSAMP +
							grad
						local densitymid = n_absmid * MIDAMP + densitybase
						local canexp = 0.5 + terblen * 0.5
						local canamp = terblen * CANAMP
						local density = n_absterrain ^ canexp * canamp * n_absmid +
							densitymid
						
						if y >= 1 and density > -0.005 and density < 0 then
							ysp = y + 1
							xsp = x
							zsp = z
							break
						end
						nixz = nixz + 1
						nixyz = nixyz + 1
					end
					if ysp then
						break
					end
					nixz = nixz - 80
				end
				if ysp then
					break
				end
				nixz = nixz + 80
			end
			if ysp then
				break
			end
		end
		print ("[watershed] spawn player (" .. xsp .. " " .. ysp .. " " .. zsp .. ")")
		player:setpos({x = xsp, y = ysp, z = zsp})
	end

	minetest.register_on_newplayer(function(player)
		spawnplayer(player)
	end)

	minetest.register_on_respawnplayer(function(player)
		spawnplayer(player)
		return true
	end)
end
