local storage = core.get_mod_storage()
local OLLAMA_URL = storage:get_string("ollama_url")
if OLLAMA_URL == "" then OLLAMA_URL = "http://127.0.0.1:11434"; storage:set_string("ollama_url", OLLAMA_URL) end
local MODEL = storage:get_string("ollama_model")
if MODEL == "" then MODEL = "gemma3:4b"; storage:set_string("ollama_model", MODEL) end
local API_URL = OLLAMA_URL .. "/api/generate"
core.log("action", ("[ServerHealth] Using Ollama model '%s' at %s"):format(MODEL, API_URL))

local API_KEY = storage:get_string("api_key") -- unused now

local http_api = core.request_http_api and core.request_http_api()
if not http_api then
  core.log("warning", "[ServerHealth] HTTP API unavailable. AI disabled.")
end

local track = { chat = {}, nodes = {}, flagged_chats = {}, flagged_nodes = {} }
local ui = {}
local timestamp = os.date

local function parse_json_safe(s)
  if type(s) ~= "string" then return nil end
  local ok, result = pcall(core.parse_json, s, nil, true)
  return ok and result or nil
end

local function write_json_safe(t)
  if type(t) ~= "table" then return nil end
  local ok, result = pcall(core.write_json, t, false)
  return ok and result or nil
end

local function make_ollama_body(prompt)
  return write_json_safe({
    model = MODEL,
    prompt = prompt,
    stream = false
  })
end

local function notify_admins(msg)
  for _, player_obj in ipairs(core.get_connected_players()) do
    local player_name = player_obj:get_player_name()
    if core.check_player_privs(player_name, { server = true }) then
      local hud_id = player_obj:hud_add{
        hud_elem_type = "text",
        position = { x = 0.5, y = 0.1 },
        text = core.formspec_escape(msg),
        number = 0xFF4500,
        alignment = { x = 0, y = 0 },
        style = 1
      }
      core.after(7, function()
        if player_obj and player_obj:is_player() then
          player_obj:hud_remove(hud_id)
        end
      end)
    end
  end
end

local SEVERITY_THRESHOLD = 5

local function analyze_chat_message(message, player_name)
  if not http_api then return end

  local body = make_ollama_body(('Analyze this chat. Return severity 0–10 and a reason.\nMessage: "%s"'):format(message))
  if not body then return end

  http_api.fetch({
    url = API_URL,
    method = "POST",
    data = body,
    timeout = 120,
    extra_headers = { "Content-Type: application/json" }
  }, function(res)
    local parsed = res and res.succeeded and parse_json_safe(res.data)
    local text = parsed and parsed.response
    local severity = text and tonumber(text:match("%d+"))
    if severity and severity > SEVERITY_THRESHOLD then
      table.insert(track.flagged_chats, {
        timestamp = timestamp(),
        player = player_name,
        message = message,
        analysis = text,
        severity = severity
      })
      notify_admins(("[AI] Flagged chat from %s (Severity: %d): \"%s\""):format(player_name, severity, message))
    end
  end)
end

local function ask_ai_for_player(player_name, query)
  if not http_api then
    core.chat_send_player(player_name, "ServerHealth: AI unavailable.")
    return
  end

  local body = make_ollama_body(('Make response compact, add reason & message to violator:\n%s'):format(query))
  if not body then return end

  http_api.fetch({
    url = API_URL,
    method = "POST",
    data = body,
    timeout = 120,
    extra_headers = { "Content-Type: application/json" }
  }, function(res)
    local response_text = "(no response)"
    if res and res.succeeded then
      local parsed = parse_json_safe(res.data)
      response_text = parsed and parsed.response or response_text
    end
    core.chat_send_player(player_name, "ServerHealth AI: " .. response_text)
  end)
end

core.register_on_chat_message(function(name, message)
  table.insert(track.chat, { player = name, message = message, timestamp = timestamp() })
  analyze_chat_message(message, name)
  return false
end)

core.register_on_placenode(function(pos, newnode, placer)
  local name = placer and placer.is_player and placer:get_player_name() or "unknown"
  table.insert(track.nodes, {
    player = name,
    node = newnode.name,
    pos = { x = pos.x, y = pos.y, z = pos.z },
    timestamp = timestamp()
  })

  if newnode.name == "default:lava_source" or newnode.name == "default:water_source" then
    local message = "Placed forbidden node: " .. newnode.name .. (" at (%d,%d,%d)"):format(pos.x, pos.y, pos.z)
    table.insert(track.flagged_nodes, {
      timestamp = timestamp(),
      player = name,
      node = newnode.name,
      pos = pos,
      message = message,
      severity = 10
    })
    notify_admins(("[ALERT] %s placed forbidden node: %s"):format(name, newnode.name))
  end
end)

if rawget(_G, "bucket") and bucket.registered_buckets then
  local original_bucket_register = bucket.register
  bucket.register = function(name, def)
    local old_on_use = def.on_use
    def.on_use = function(itemstack, user, pointed_thing)
      if user and user:is_player() then
        local pname = user:get_player_name()
        if name:find("lava") or name:find("water") then
          table.insert(track.flagged_nodes, {
            timestamp = timestamp(),
            player = pname,
            node = name,
            message = "Used bucket: " .. name,
            severity = 10
          })
          notify_admins(("[ALERT] %s used forbidden bucket: %s"):format(pname, name))
        end
      end
      if old_on_use then
        return old_on_use(itemstack, user, pointed_thing)
      end
      return itemstack
    end
    original_bucket_register(name, def)
  end
end


core.register_chatcommand("setmodel", {
  params = "<ollama_model>",
  privs = { server = true },
  func = function(name, model)
    if not model or model == "" then
      return false, "Provide an AI model name."
    end
    MODEL = model
    storage:set_string("ollama_model", model)
    core.chat_send_player(name, "[ServerHealth] Model set to: " .. model)
    return true
  end
})

local function build_list_entries(list)
  local entries = {}
  for _, e in ipairs(list) do
    table.insert(entries, core.formspec_escape(("%s | %s | \"%s\" (sev %d)"):format(e.timestamp, e.player, e.message or e.node, e.severity or 0)))
  end
  return table.concat(entries, ",")
end

local function get_summary(callback)
  if not http_api then
    return callback("(AI unavailable)")
  end

  local combined = {}
  for _, e in ipairs(track.flagged_chats) do
    table.insert(combined, ("[Chat] %s | %s | %s (sev %d)"):format(e.timestamp, e.player, e.message or "", e.severity or 0))
  end
  for _, e in ipairs(track.flagged_nodes) do
    table.insert(combined, ("[Node] %s | %s | %s (sev %d)"):format(e.timestamp, e.player, e.node or "", e.severity or 0))
  end

  if #combined > 50 then
    local trimmed = {}
    for i = 1, 50 do
      table.insert(trimmed, combined[i])
    end
    combined = trimmed
  end

  local prompt = ("Summarize these flagged server actions briefly and clearly (max 3 sentences). Then give a server health score as a number/10 on its own line.\n\n%s"):format(table.concat(combined, "\n"))
  local body = make_ollama_body(prompt)
  if not body then
    return callback("(Failed to build AI request)")
  end

  http_api.fetch({
    url = API_URL,
    method = "POST",
    data = body,
    timeout = 120,
    extra_headers = { "Content-Type: application/json" }
  }, function(res)
    local summary = "(no response)"
    if res and res.succeeded then
      local parsed = parse_json_safe(res.data)
      summary = parsed and parsed.response or summary
    end
    callback(summary)
  end)
end

local function ui_show(player_name)
  local stats = ("Chats: %d | Nodes: %d | Flagged Chats: %d | Flagged Nodes: %d"):format(
    #track.chat, #track.nodes, #track.flagged_chats, #track.flagged_nodes
  )
  local state = ui[player_name] or { tab = "chats", selected = 0, summary = "(not generated)" }
  ui[player_name] = state

  local flagged_list = (state.tab == "chats") and track.flagged_chats or track.flagged_nodes

local forms = {
    "formspec_version[4]",
    "size[14,9.5]",
    "position[0.5,0.5]",
    "anchor[0.5,0.5]",
    "bgcolor[#2D2D2D;true]",
    "label[5,0.5;Dashboard]",
    "label[3,1.0;" .. core.formspec_escape(stats) .. "]",
    "style[chats_tab;border=false;bgcolor=" .. (state.tab == "chats" and "#5A5A5A" or "#333333") .. ";textcolor=white]",
    "button[1,2;3.5,1;chats_tab;Flagged Chats]",
    "style[nodes_tab;border=false;bgcolor=" .. (state.tab == "nodes" and "#5A5A5A" or "#333333") .. ";textcolor=white]",
    "button[5.25,2;3.5,1;nodes_tab;Flagged Nodes]",
    "style[summary_tab;border=false;bgcolor=" .. (state.tab == "summary" and "#5A5A5A" or "#333333") .. ";textcolor=white]",
    "button[9.5,2;3.5,1;summary_tab;Summary]",
    "box[1,3.2;12,5;#1E1E1E]",
}

  if state.tab == "summary" then
    table.insert(forms, "textarea[1.2,3.4;11.6,4.6;summary_box;Summary:;" .. core.formspec_escape(state.summary) .. "]")
    table.insert(forms, "button[5,8.5;4,0.8;gen_summary;Generate Summary]")
  else
    table.insert(forms, "textlist[1.2,3.4;11.6,4.6;flagged_list;" .. build_list_entries(flagged_list) .. ";" .. (state.selected or 0) .. ";true]")
    table.insert(forms, "style[action_btn;bgcolor=#444444;textcolor=white]")
    table.insert(forms, "button[1,8.5;4,0.8;refresh;Refresh]")
    table.insert(forms, "button[9,8.5;4,0.8;ask_ai;Ask AI about selected]")
  end

  core.show_formspec(player_name, "serverhealth:panel", table.concat(forms, ""))
end

core.register_on_player_receive_fields(function(player, formname, fields)
  if formname ~= "serverhealth:panel" then return end
  local player_name = player:get_player_name()
  local state = ui[player_name] or { tab = "chats", selected = 0, summary = "(not generated)" }
  ui[player_name] = state

  if fields.chats_tab then
    state.tab = "chats"
    state.selected = 0
    ui_show(player_name)
  elseif fields.nodes_tab then
    state.tab = "nodes"
    state.selected = 0
    ui_show(player_name)
  elseif fields.summary_tab then
    state.tab = "summary"
    ui_show(player_name)
  end

  if fields.flagged_list then
    local ev = core.explode_textlist_event(fields.flagged_list)
    if ev and ev.index then state.selected = ev.index end
  end

  if fields.refresh then
    ui_show(player_name)
  end

  if fields.ask_ai then
    local flagged = (state.tab == "chats") and track.flagged_chats or track.flagged_nodes
    local sel = state.selected
    if not sel or not flagged[sel] then
      core.chat_send_player(player_name, "ServerHealth: No flagged entry selected.")
      return
    end
    local entry = flagged[sel]
    ask_ai_for_player(player_name, ("Moderation advice for %s (sev %d): \"%s\""):format(entry.player, entry.severity or 0, entry.message or entry.node))
  end

  if fields.gen_summary then
    get_summary(function(summary_text)
      state.summary = summary_text or "(no response)"
      ui_show(player_name)
    end)
  end
end)

core.register_chatcommand("dashboard", {
  description = "Open the ServerHealth dashboard.",
  privs = { server = true },
  func = function(name)
    ui_show(name)
    return true
  end
})

local function chatbot_query(player_name, query)
  if not http_api then
    core.chat_send_player(player_name, "ServerHealth: AI unavailable.")
    return
  end

  local context = {}
  for _, e in ipairs(track.flagged_chats) do
    table.insert(context, ("[Chat] %s | %s | \"%s\" (sev %d)"):format(e.timestamp, e.player, e.message or "", e.severity or 0))
  end
  for _, e in ipairs(track.flagged_nodes) do
    table.insert(context, ("[Node] %s | %s | %s (sev %d)"):format(e.timestamp, e.player, e.node or "", e.severity or 0))
  end

  local prompt = ("Context:\n%s\n\nQuestion: %s"):format(table.concat(context, "\n"), query)
  local body = make_ollama_body(('You are the assistant in a Luanti (formerly Minetest) game server. Make your response useful to the server admin. Also keep your response compact and in one line.\n%s'):format(prompt))
  if not body then return end

  http_api.fetch({
    url = API_URL,
    method = "POST",
    data = body,
    timeout = 120,
    extra_headers = { "Content-Type: application/json" }
  }, function(res)
    local response_text = "(no response)"
    if res and res.succeeded then
      local parsed = parse_json_safe(res.data)
      response_text = parsed and parsed.response or response_text
    end
    core.chat_send_player(player_name, "ServerHealth AI: " .. response_text)
  end)
end

core.register_chatcommand("chatbot", {
  description = "Ask the ServerHealth AI about flagged messages or nodes.",
  privs = { server = true },
  params = "<query>",
  func = function(name, query)
    if not query or query:match("^%s*$") then
      return false, "Provide a query."
    end
    chatbot_query(name, query)
    return true
  end
})
