import type { MtVec2, MtVec3 } from "@repcomm/mt-api";
import type { NodeData, VoxelManip } from "./voxel_manip";
import type { Vector } from "./vector";

let modname = core.get_current_modname();
let modpath = core.get_modpath(modname);

if (core.get_modpath("megamorph") != undefined) {
    megamorph.register_loot({ name: "advtrains:engine_japan", level: 1, rarity: 7, number: { min: 1, max: 1 } })
    const old_write_to_map: (this: any, ...rest: any[]) => any = Geomorph.get("write_to_map")
    Geomorph.set("write_to_map", function (this: any, ...rest: any[]) {
        const station = Station.get(vector.multiply(vector.add(this.minp, this.maxp), 0.5));
        if (station != undefined && station.draws_in(this.minp, this.maxp))
            return;
        return old_write_to_map.call(this, ...rest)
    })
}

advtrains.wagon_prototypes["advtrains:engine_japan"].drops = ["advtrains:engine_japan"];

core.set_mapgen_setting("chunksize", 5, true)

const CHUNK_SIZE = 80;
const CHUNK_OFFSET = -32;
const BIG_CHUNK_SIZE = 25 * CHUNK_SIZE; // 2000
const LOW_RAIL_CHUNKS = 12;
//const BIG_CHUNK_SIZE = 4 * CHUNK_SIZE; // for debugging
//const LOW_RAIL_CHUNKS = 3; // for debugging

enum RailLevel {
    Low = -LOW_RAIL_CHUNKS * CHUNK_SIZE + CHUNK_OFFSET,
    High = -2 * CHUNK_SIZE + CHUNK_OFFSET,
}

const KNOWN_RAIL_LEVELS: RailLevel[] = [RailLevel.Low, RailLevel.High];

const CENTRAL_EMPTY_ZONE = 512;

function get_rail_level(x: number, z: number): undefined | RailLevel {
    if (x >= -CENTRAL_EMPTY_ZONE && x <= CENTRAL_EMPTY_ZONE && z >= -CENTRAL_EMPTY_ZONE && z <= CENTRAL_EMPTY_ZONE) {
        return undefined;
    }
    let retval;
    if (false) {
        // for debugging
        if (x ** 2 + z ** 2 <= BIG_CHUNK_SIZE * BIG_CHUNK_SIZE) {
            retval = RailLevel.High;
        } else {
            retval = RailLevel.Low;
        }
    } else {
        const r = new PcgRandom(x * 923284 + z * 263417623);
        if (r.next(0, 5) <= 3) {
            const index = r.next(0, KNOWN_RAIL_LEVELS.length - 1);
            retval = KNOWN_RAIL_LEVELS[index];
        } else {
            retval = undefined;
        }
    }
    // core.log("none", `get_rail_level(${x}, ${z}) => ${retval}`);
    return retval;
}

const CIRCLE_CACHE: LuaMap<number, MtVec2[]> = new LuaMap()

function make_circle(radius: number): MtVec2[] {
    if (!(radius > 0)) { // catch NaN too
        return [{ x: 0, y: 0 }];
    }
    const radius_sq = radius * radius;
    const retval: MtVec2[] = [];
    let x = 0, y = math.floor(radius);
    while (y >= x) {
        retval.push({ x: x, y: y }, { x: y, y: -x }, { x: -x, y: -y }, { x: -y, y: x });
        if (y > x) {
            retval.push({ x: -x, y: y }, { x: -y, y: -x }, { x: x, y: -y }, { x: y, y: x });
        }
        x++;
        if ((x * x) + (y * y) > radius_sq) {
            y--;
        }
    }
    return retval;
}

function get_circle(radius: number): MtVec2[] {
    let retval = CIRCLE_CACHE.get(radius);
    if (retval != undefined) return retval;
    retval = make_circle(radius);
    CIRCLE_CACHE.set(radius, retval);
    return retval;
}

enum XZRot {
    Rot0 = 0,
    Rot90 = 1,
    Rot180 = 2,
    Rot270 = 3,
}

function xz_rot_reverse(v: XZRot): XZRot {
    switch (v) {
        case XZRot.Rot0:
            return XZRot.Rot180;
        case XZRot.Rot90:
            return XZRot.Rot270;
        case XZRot.Rot180:
            return XZRot.Rot0;
        case XZRot.Rot270:
            return XZRot.Rot90;
    }
}

const XZ_ROT_VALUES: XZRot[] = [XZRot.Rot0, XZRot.Rot90, XZRot.Rot180, XZRot.Rot270];

function xz_rot(pos: MtVec3, center?: MtVec3, angle?: XZRot): MtVec3 {
    if (angle == undefined || center == undefined) { return pos; }
    switch (angle) {
        case XZRot.Rot0:
            return pos;
        case XZRot.Rot90:
            return vector.new(center.x + (pos.z - center.z), pos.y, center.z - (pos.x - center.x));
        case XZRot.Rot180:
            return vector.new(center.x - (pos.x - center.x), pos.y, center.z - (pos.z - center.z));
        case XZRot.Rot270:
            return vector.new(center.x - (pos.z - center.z), pos.y, center.z + (pos.x - center.x));
    }
}

function xz_rot_cube(minp: MtVec3, maxp: MtVec3, center?: MtVec3, angle?: XZRot): LuaMultiReturn<[MtVec3, MtVec3]> {
    if (angle == undefined || center == undefined) { return $multi(minp, maxp); }
    switch (angle) {
        case XZRot.Rot0:
            return $multi(minp, maxp);
        case XZRot.Rot90:
            return $multi(
                vector.new(center.x + (minp.z - center.z), minp.y, center.z - (maxp.x - center.x)),
                vector.new(center.x + (maxp.z - center.z), maxp.y, center.z - (minp.x - center.x)));
        case XZRot.Rot180:
            return $multi(
                vector.new(center.x - (maxp.x - center.x), minp.y, center.z - (maxp.z - center.z)),
                vector.new(center.x - (minp.x - center.x), maxp.y, center.z - (minp.z - center.z)));
        case XZRot.Rot270:
            return $multi(
                vector.new(center.x - (maxp.z - center.z), minp.y, center.z + (minp.x - center.x)),
                vector.new(center.x - (minp.z - center.z), maxp.y, center.z + (maxp.x - center.x)));
    }
}

class Drawer {
    vm: VoxelManip | undefined;
    drew_anything: boolean;
    constructor(vm: VoxelManip | undefined) {
        this.vm = vm;
        this.drew_anything = false;
    }
    node(pos: MtVec3, node: NodeData, rot_center?: MtVec3, rot_angle?: XZRot) {
        this.drew_anything = true;
        const vm = this.vm;
        if (vm == undefined) return;
        pos = xz_rot(pos, rot_center, rot_angle);
        vm.set_node_at(pos, node);
    }
    cube(minp: MtVec3, maxp: MtVec3, node: NodeData, step?: number): void;
    cube(minp: MtVec3, maxp: MtVec3, node: NodeData, rot_center: MtVec3, rot_angle: XZRot, step?: number): void;
    cube(minp: MtVec3, maxp: MtVec3, node: NodeData, rot_center_or_step?: MtVec3 | number, rot_angle?: XZRot, step?: number) {
        this.drew_anything = true;
        const vm = this.vm;
        if (vm == undefined) return;
        let rot_center: MtVec3 | undefined = undefined;
        if (rot_center_or_step == undefined) {
            step = 1;
        } else if (typeof rot_center_or_step == "number") {
            step = rot_center_or_step;
        } else {
            rot_center = rot_center_or_step;
            if (step == undefined) {
                step = 1;
            }
        }
        [minp, maxp] = xz_rot_cube(minp, maxp, rot_center, rot_angle);
        const pos = vector.zero();
        for (const z of $range(minp.z, maxp.z, step)) {
            pos.z = z;
            for (const y of $range(minp.y, maxp.y, step)) {
                pos.y = y;
                for (const x of $range(minp.x, maxp.x, step)) {
                    pos.x = x;
                    vm.set_node_at(pos, node)
                }
            }
        }
    }
    circle(center: MtVec3, radius: number, plane: ["x", "y"] | ["x", "z"] | ["y", "z"], node: NodeData) {
        this.drew_anything = true;
        const vm = this.vm;
        if (vm == undefined) return;
        center = vector.floor(vector.add(center, 0.5));
        const pos = vector.copy(center);
        const circle = get_circle(radius);
        for (const { x, y } of circle) {
            pos[plane[0]] = center[plane[0]] + x
            pos[plane[1]] = center[plane[1]] + y
            vm.set_node_at(pos, node)
        }
    }
    line(start: MtVec3, inc: { x?: number, y?: number, z?: number }, count: number, node: NodeData, rot_center?: MtVec3, rot_angle?: XZRot) {
        this.drew_anything = true;
        const vm = this.vm;
        if (vm == undefined) return;
        const pos = vector.copy(xz_rot(start, rot_center, rot_angle));
        const inc2 = xz_rot({
            x: inc.x != undefined ? inc.x : 0,
            y: inc.y != undefined ? inc.y : 0,
            z: inc.z != undefined ? inc.z : 0,
        }, { x: 0, y: 0, z: 0 }, rot_angle);
        for (const _ of $range(1, count)) {
            vm.set_node_at(pos, node)
            pos.x += inc2.x;
            pos.y += inc2.y;
            pos.z += inc2.z;
        }
    }
}

function minp_size_to_minp_maxp(minp: MtVec3, size: MtVec3): [Vector, Vector] {
    return [vector.copy(minp), vector.add(vector.add(minp, size), -1)]
}

const TRACK_STEP_SIZE = 2;

enum TrackYStep {
    Down = -1,
    Same = 0,
    Up = 1,
}

const TRACK_Y_STEPS = [TrackYStep.Down, TrackYStep.Same, TrackYStep.Up];

function make_track_nodes(): { [K in TrackYStep]: { [K2 in XZRot]: NodeData[] } } {
    let retval: { [K in TrackYStep]: { [K2 in XZRot]?: NodeData[] } } = {
        [TrackYStep.Down]: {},
        [TrackYStep.Same]: {},
        [TrackYStep.Up]: {},
    };
    for (const xz_rot of XZ_ROT_VALUES) {
        let up_param2: number, same_param2: number, down_param2: number;
        switch (xz_rot) {
            case XZRot.Rot0:
                up_param2 = 1;
                same_param2 = 3;
                down_param2 = 3;
                break;
            case XZRot.Rot90:
                up_param2 = 2;
                same_param2 = 2;
                down_param2 = 0;
                break;
            case XZRot.Rot180:
                up_param2 = 3;
                same_param2 = 3;
                down_param2 = 1;
                break;
            case XZRot.Rot270:
                up_param2 = 0;
                same_param2 = 2;
                down_param2 = 2;
                break;
        }
        retval[TrackYStep.Down][xz_rot] = [{
            name: "advtrains:dtrack_vst2",
            param1: 0,
            param2: down_param2,
        }, {
            name: "advtrains:dtrack_vst1",
            param1: 0,
            param2: down_param2,
        }];
        const same_node = {
            name: "advtrains:dtrack_st",
            param1: 0,
            param2: same_param2,
        };
        retval[TrackYStep.Same][xz_rot] = [same_node, same_node];
        retval[TrackYStep.Up][xz_rot] = [{
            name: "advtrains:dtrack_vst1",
            param1: 0,
            param2: up_param2,
        }, {
            name: "advtrains:dtrack_vst2",
            param1: 0,
            param2: up_param2,
        }];
        for (const track_y_step of TRACK_Y_STEPS) {
            assert(retval[track_y_step][xz_rot]!.length == TRACK_STEP_SIZE);
        }
    }
    return (retval as { [K in TrackYStep]: { [K2 in XZRot]: NodeData[] } });
}

const TRACK_NODES = make_track_nodes();

const TRACK_CEILING_Y = 9;
const TRACK_FLOOR_Y = -1;
const TRACK_WALL_MIN_Z = -3;
const TRACK_WALL_MAX_Z = 3;
const TRACK_WALL_LEN = BIG_CHUNK_SIZE - CHUNK_SIZE + 1;
const TRACK_EXTRA_LEN = 10;

type MtVec3Set = LuaMap<number, LuaMap<number, LuaSet<number>>>;

/** inserts `key` into `set`, returns `true` if `key` was successfully inserted (was not present beforehand). */
function try_insert_in_mt_vec3_set(set: MtVec3Set, key: MtVec3): boolean {
    let yz = set.get(key.x);
    if (yz == undefined) {
        set.set(key.x, yz = new LuaMap());
    }
    let z = yz.get(key.y);
    if (z == undefined) {
        yz.set(key.y, z = new LuaSet());
    }
    const retval = !z.has(key.z);
    z.add(key.z);
    return retval;
}

class Station {
    pos: Vector;
    rail_level: RailLevel;
    _neighbors?: Station[];
    minp: Vector;
    maxp: Vector;
    get floor_minp(): Vector {
        return vector.add(this.inner_minp, -1);
    }
    get floor_maxp(): Vector {
        const inner_minp = this.inner_minp;
        const inner_maxp = this.inner_maxp;
        return vector.new(inner_maxp.x + 1, inner_minp.y - 1, inner_maxp.z + 1);
    }
    get ceiling_minp(): Vector {
        const inner_minp = this.inner_minp;
        const inner_maxp = this.inner_maxp;
        return vector.new(inner_minp.x - 1, inner_maxp.y + 1, inner_minp.z - 1);
    }
    get ceiling_maxp(): Vector {
        return vector.add(this.inner_maxp, 1);
    }
    get inner_minp(): Vector {
        return vector.offset(this.pos, 1, 21, 1);
    }
    get inner_floor_midp(): Vector {
        return vector.offset(this.pos, 40, 21, 40);
    }
    get inner_maxp(): Vector {
        return vector.offset(this.pos, 78, 53, 78);
    }
    get outer_minp(): Vector {
        return vector.add(this.inner_minp, -1);
    }
    get outer_maxp(): Vector {
        return vector.add(this.inner_maxp, 1);
    }
    get track_starts(): [Vector, Vector] {
        return [vector.offset(this.pos, 79, 21, 30), vector.offset(this.pos, 79, 21, 50)];
    }
    static readonly EXITS: [Vector, Vector][] = [
        minp_size_to_minp_maxp(vector.new(0, 21, 19), vector.new(1, 3, 2)),
        minp_size_to_minp_maxp(vector.new(0, 21, 39), vector.new(1, 3, 2)),
        minp_size_to_minp_maxp(vector.new(0, 21, 59), vector.new(1, 3, 2)),
        minp_size_to_minp_maxp(vector.new(79, 21, 19), vector.new(1, 3, 2)),
        minp_size_to_minp_maxp(vector.new(79, 21, 39), vector.new(1, 3, 2)),
        minp_size_to_minp_maxp(vector.new(79, 21, 59), vector.new(1, 3, 2)),
        minp_size_to_minp_maxp(vector.new(19, 21, 0), vector.new(2, 3, 1)),
        minp_size_to_minp_maxp(vector.new(39, 21, 0), vector.new(2, 3, 1)),
        minp_size_to_minp_maxp(vector.new(59, 21, 0), vector.new(2, 3, 1)),
        minp_size_to_minp_maxp(vector.new(19, 21, 79), vector.new(2, 3, 1)),
        minp_size_to_minp_maxp(vector.new(39, 21, 79), vector.new(2, 3, 1)),
        minp_size_to_minp_maxp(vector.new(59, 21, 79), vector.new(2, 3, 1)),
        minp_size_to_minp_maxp(vector.new(0, 51, 39), vector.new(1, 3, 2)),
        minp_size_to_minp_maxp(vector.new(79, 51, 39), vector.new(1, 3, 2)),
        minp_size_to_minp_maxp(vector.new(39, 51, 0), vector.new(2, 3, 1)),
        minp_size_to_minp_maxp(vector.new(39, 51, 79), vector.new(2, 3, 1)),
    ];
    static readonly EXIT_TUNNELS: [Vector, Vector][] = [
        minp_size_to_minp_maxp(vector.new(0, 21, 19), vector.new(80, 3, 2)),
        minp_size_to_minp_maxp(vector.new(0, 21, 39), vector.new(80, 3, 2)),
        minp_size_to_minp_maxp(vector.new(0, 21, 59), vector.new(80, 3, 2)),
        minp_size_to_minp_maxp(vector.new(19, 21, 0), vector.new(2, 3, 80)),
        minp_size_to_minp_maxp(vector.new(39, 21, 0), vector.new(2, 3, 80)),
        minp_size_to_minp_maxp(vector.new(59, 21, 0), vector.new(2, 3, 80)),
        minp_size_to_minp_maxp(vector.new(0, 51, 39), vector.new(80, 3, 2)),
        minp_size_to_minp_maxp(vector.new(39, 51, 0), vector.new(2, 3, 80)),
    ];
    static readonly WALLS: [Vector, Vector][] = [
        minp_size_to_minp_maxp(vector.new(0, 21, 0), vector.new(1, 34, 80)),
        minp_size_to_minp_maxp(vector.new(0, 21, 0), vector.new(80, 34, 1)),
        minp_size_to_minp_maxp(vector.new(79, 21, 0), vector.new(1, 34, 80)),
        minp_size_to_minp_maxp(vector.new(0, 21, 79), vector.new(80, 34, 1)),
    ];
    get neighbors(): Station[] {
        if (this._neighbors == undefined) {
            this._neighbors = [];
            for (const pos of [
                vector.offset(this.pos, -BIG_CHUNK_SIZE, 0, 0),
                vector.offset(this.pos, BIG_CHUNK_SIZE, 0, 0),
                vector.offset(this.pos, 0, 0, -BIG_CHUNK_SIZE),
                vector.offset(this.pos, 0, 0, BIG_CHUNK_SIZE),
            ]) {
                const rail_level = get_rail_level(pos.x, pos.z)
                if (rail_level != undefined) {
                    this._neighbors.push(new Station(pos, rail_level));
                }
            }
        }
        return this._neighbors;
    }
    static get(input_pos: MtVec3): undefined | Station {
        let pos = vector.multiply(vector.subtract(input_pos, CHUNK_OFFSET), 1 / BIG_CHUNK_SIZE)
        pos.y = 0
        pos = vector.add(vector.multiply(vector.floor(vector.add(pos, 0.5)), BIG_CHUNK_SIZE), CHUNK_OFFSET)
        const rail_level = get_rail_level(pos.x, pos.z)
        if (rail_level != undefined) {
            return new Station(pos, rail_level);
        } else {
            return undefined;
        }
    }
    constructor(pos: Vector, rail_level: RailLevel) {
        this.pos = pos;
        pos.y = rail_level;
        this.rail_level = rail_level;
        this.minp = vector.offset(pos, 0, 0, 0)
        this.maxp = vector.offset(pos, CHUNK_SIZE - 1, CHUNK_SIZE - 1, CHUNK_SIZE - 1)
    }
    draw_track_step(
        neighbor: Station,
        drawer: Drawer,
        minp: MtVec3,
        maxp: MtVec3,
        pos: Vector,
        rot_center: Vector,
        rot_angle: XZRot,
        track_y_step: TrackYStep,
    ): void {
        const [step_minp, step_maxp] = xz_rot_cube(
            vector.offset(pos, 0, TRACK_FLOOR_Y, TRACK_WALL_MIN_Z),
            vector.offset(pos, TRACK_STEP_SIZE - 1, TRACK_CEILING_Y, TRACK_WALL_MAX_Z),
            rot_center, rot_angle);
        const [iminp, imaxp] = intersect_cubes(minp, maxp, step_minp, step_maxp);
        if (iminp == undefined) { return; }
        const track_nodes = TRACK_NODES[track_y_step][rot_angle];
        for (let i of $range(0, TRACK_STEP_SIZE - 1)) {
            drawer.node(vector.offset(pos, i, 0, 0), track_nodes[i], rot_center, rot_angle);
        }
    }
    draw_track_walls_step(
        neighbor: Station,
        drawer: Drawer,
        minp: MtVec3,
        maxp: MtVec3,
        pos: Vector,
        rot_center: Vector,
        rot_angle: XZRot,
    ): void {
        const [step_minp, step_maxp] = xz_rot_cube(
            vector.offset(pos, 0, TRACK_FLOOR_Y, TRACK_WALL_MIN_Z),
            vector.offset(pos, TRACK_STEP_SIZE - 1, TRACK_CEILING_Y, TRACK_WALL_MAX_Z),
            rot_center, rot_angle);
        const [iminp, imaxp] = intersect_cubes(minp, maxp, step_minp, step_maxp);
        if (iminp == undefined) { return; }
        drawer.cube(vector.offset(pos, 0, TRACK_FLOOR_Y, TRACK_WALL_MIN_Z),
            vector.offset(pos, TRACK_STEP_SIZE - 1, TRACK_CEILING_Y, TRACK_WALL_MIN_Z),
            {
                name: "artifacts:moon_glass",
                param1: 0,
                param2: 0
            }, rot_center, rot_angle);
        drawer.cube(vector.offset(pos, 0, TRACK_FLOOR_Y, TRACK_WALL_MAX_Z),
            vector.offset(pos, TRACK_STEP_SIZE - 1, TRACK_CEILING_Y, TRACK_WALL_MAX_Z),
            {
                name: "artifacts:moon_glass",
                param1: 0,
                param2: 0
            }, rot_center, rot_angle);
        drawer.cube(vector.offset(pos, 0, TRACK_FLOOR_Y, TRACK_WALL_MIN_Z),
            vector.offset(pos, TRACK_STEP_SIZE - 1, TRACK_FLOOR_Y, TRACK_WALL_MAX_Z),
            {
                name: "artifacts:moon_glass",
                param1: 0,
                param2: 0
            }, rot_center, rot_angle);
        drawer.cube(vector.offset(pos, 0, TRACK_CEILING_Y, TRACK_WALL_MIN_Z),
            vector.offset(pos, TRACK_STEP_SIZE - 1, TRACK_CEILING_Y, TRACK_WALL_MAX_Z),
            {
                name: "artifacts:moon_glass",
                param1: 0,
                param2: 0
            }, rot_center, rot_angle);
        drawer.cube(vector.offset(pos, 0, TRACK_FLOOR_Y + 1, TRACK_WALL_MIN_Z + 1),
            vector.offset(pos, TRACK_STEP_SIZE - 1, TRACK_CEILING_Y - 1, TRACK_WALL_MAX_Z - 1),
            {
                name: "air",
                param1: 0,
                param2: 0
            }, rot_center, rot_angle);
    }
    draw_track(neighbor: Station, drawer: Drawer, minp: MtVec3, maxp: MtVec3, exit_tunnels_not_needed: MtVec3Set) {
        let rot_angle: XZRot;
        if (neighbor.pos.x > this.pos.x) {
            rot_angle = XZRot.Rot0;
        } else if (neighbor.pos.x < this.pos.x) {
            rot_angle = XZRot.Rot180;
            assert(false, `this = ${this.pos}, neighbor = ${neighbor.pos}`);
        } else if (neighbor.pos.z < this.pos.z) {
            rot_angle = XZRot.Rot90;
            assert(false, `this = ${this.pos}, neighbor = ${neighbor.pos}`);
        } else {
            rot_angle = XZRot.Rot270;
        };
        const rot_center = this.inner_floor_midp;
        for (const track_index of $range(0, 1)) {
            let track_y_step: TrackYStep = TrackYStep.Same;
            if (this.pos.y > neighbor.pos.y) {
                track_y_step = TrackYStep.Down;
            } else if (this.pos.y < neighbor.pos.y) {
                track_y_step = TrackYStep.Up;
            }
            let wall_track_y_step = track_y_step;
            const target_y = neighbor.track_starts[track_index].y;
            let pos = vector.copy(this.track_starts[track_index]);
            for (const i of $range(1, TRACK_WALL_LEN, TRACK_STEP_SIZE)) {
                if (pos.y == target_y) {
                    wall_track_y_step = TrackYStep.Same;
                }
                this.draw_track_walls_step(neighbor, drawer, minp, maxp, pos, rot_center, rot_angle);
                pos.x += TRACK_STEP_SIZE;
                pos.y += wall_track_y_step;
                if (track_index == 0) {
                    const final_pos = xz_rot(pos, rot_center, rot_angle);
                    const chunk_index = vector.floor(vector.multiply(vector.subtract(final_pos, CHUNK_OFFSET - 0.001), 1 / CHUNK_SIZE));
                    const chunk_minp = vector.add(vector.multiply(chunk_index, CHUNK_SIZE), CHUNK_OFFSET);
                    this.draw_exit_tunnels(drawer, minp, maxp, chunk_minp, exit_tunnels_not_needed);
                }
            }
            pos = vector.copy(this.track_starts[track_index]);
            for (const i of $range(1, TRACK_WALL_LEN + TRACK_EXTRA_LEN, TRACK_STEP_SIZE)) {
                let draw_track_y_step = track_y_step;
                if (pos.y == target_y) {
                    track_y_step = TrackYStep.Same;
                }
                if (draw_track_y_step == TrackYStep.Down) {
                    if (i == 1) draw_track_y_step = TrackYStep.Same;
                } else {
                    draw_track_y_step = track_y_step;
                }
                this.draw_track_step(neighbor, drawer, minp, maxp, pos, rot_center, rot_angle, draw_track_y_step);
                pos.x += TRACK_STEP_SIZE;
                pos.y += track_y_step;
            }
            pos = vector.copy(this.track_starts[track_index]);
            for (const i of $range(1, TRACK_EXTRA_LEN, TRACK_STEP_SIZE)) {
                pos.x -= TRACK_STEP_SIZE;
                this.draw_track_step(neighbor, drawer, minp, maxp, pos, rot_center, rot_angle, TrackYStep.Same);
            }
        }
    }
    draw_exit_tunnels(drawer: Drawer, minp: MtVec3, maxp: MtVec3, chunk_minp: Vector, exit_tunnels_not_needed: MtVec3Set) {
        if (!try_insert_in_mt_vec3_set(exit_tunnels_not_needed, chunk_minp)) { return; }
        const [iminp, imaxp] = intersect_cubes(minp, maxp, chunk_minp, vector.add(chunk_minp, CHUNK_SIZE - 1));
        if (iminp == undefined) { return; }
        for (const [cube_minp, cube_maxp] of Station.EXIT_TUNNELS) {
            drawer.cube(vector.add(chunk_minp, cube_minp), vector.add(chunk_minp, cube_maxp), {
                name: "air",
                param1: 0,
                param2: 0
            })
        }
    }
    draw_station(drawer: Drawer, minp: MtVec3, maxp: MtVec3, exit_tunnels_not_needed: MtVec3Set) {
        try_insert_in_mt_vec3_set(exit_tunnels_not_needed, this.pos);
        const [iminp, imaxp] = intersect_cubes(minp, maxp, this.minp, this.maxp)
        // core.log("none", `generating station at ${this.pos.x},${this.pos.y},${this.pos.z} with minp=${minp.x},${minp.y},${minp.z}  maxp=${maxp.x},${maxp.y},${maxp.z}`)
        if (iminp == undefined) { return; }
        for (const [exit_minp, exit_maxp] of Station.WALLS) {
            drawer.cube(vector.add(this.pos, exit_minp), vector.add(this.pos, exit_maxp), {
                name: "artifacts:antiquorium",
                param1: 0,
                param2: 0
            })
        }
        for (const [exit_minp, exit_maxp] of Station.EXITS) {
            drawer.cube(vector.add(this.pos, exit_minp), vector.add(this.pos, exit_maxp), {
                name: "air",
                param1: 0,
                param2: 0
            })
        }
        drawer.cube(this.ceiling_minp, this.ceiling_maxp, {
            name: "artifacts:moon_glass",
            param1: 0,
            param2: 0
        })
        drawer.cube(this.floor_minp, this.floor_maxp, {
            name: "artifacts:antiquorium",
            param1: 0,
            param2: 0
        })
        drawer.cube(this.inner_minp, this.inner_maxp, {
            name: "air",
            param1: 0,
            param2: 0
        })
        for (const radius of $range(5, 30, 4)) {
            drawer.circle(vector.offset(this.inner_floor_midp, 0, -1, 0), radius - 1e-6, ["x", "z"], {
                name: "artifacts:moon_glass",
                param1: 0,
                param2: 1
            })
        }
        const rot_center = this.inner_floor_midp;
        for (const rot_angle of XZ_ROT_VALUES) {
            drawer.line(vector.offset(this.pos, 10, 21, 30), { x: 1 }, 61, {
                name: "advtrains:dtrack_st",
                param1: 0,
                param2: (rot_angle == XZRot.Rot0 || rot_angle == XZRot.Rot180) ? 3 : 2,
            }, rot_center, rot_angle)
        }
        for (const rot_angle of XZ_ROT_VALUES) {
            drawer.node(vector.offset(this.pos, 30, 21, 30), {
                name: "advtrains:dtrack_xing_st",
                param1: 0,
                param2: 0
            }, rot_center, rot_angle)
        }
    }
    draw_all(drawer: Drawer, minp: MtVec3, maxp: MtVec3) {
        const exit_tunnels_not_needed: MtVec3Set = new LuaMap();
        this.draw_station(drawer, minp, maxp, exit_tunnels_not_needed)
        for (const neighbor of this.neighbors) {
            neighbor.draw_station(drawer, minp, maxp, exit_tunnels_not_needed)
        }
        for (const neighbor of this.neighbors) {
            if (this.pos.x < neighbor.pos.x || (this.pos.x == neighbor.pos.x && this.pos.z < neighbor.pos.z)) {
                this.draw_track(neighbor, drawer, minp, maxp, exit_tunnels_not_needed)
            } else {
                neighbor.draw_track(this, drawer, minp, maxp, exit_tunnels_not_needed)
            }
        }
    }
    draws_in(minp: MtVec3, maxp: MtVec3): boolean {
        const drawer = new Drawer(undefined)
        this.draw_all(drawer, minp, maxp)
        return drawer.drew_anything;
    }
}

function intersect_cubes(minp1: MtVec3, maxp1: MtVec3, minp2: MtVec3, maxp2: MtVec3): LuaMultiReturn<[Vector, Vector] | [undefined]> {
    const minp = vector.new(math.max(minp1.x, minp2.x), math.max(minp1.y, minp2.y), math.max(minp1.z, minp2.z))
    const maxp = vector.new(math.min(maxp1.x, maxp2.x), math.min(maxp1.y, maxp2.y), math.min(maxp1.z, maxp2.z))
    if (maxp.x >= minp.x && maxp.y >= minp.y && maxp.z >= minp.z) {
        return $multi(minp, maxp);
    }
    return $multi(undefined);
}

function lexicographically_less(lhs: MtVec3, rhs: MtVec3): boolean {
    if (lhs.x < rhs.x) return true;
    if (lhs.x != rhs.x) return false;
    if (lhs.y < rhs.y) return true;
    if (lhs.y != rhs.y) return false;
    return lhs.z < rhs.z;
}

function generate(this: void, minp: MtVec3 | undefined, maxp: MtVec3 | undefined, blockseed: any): void {
    if (!(minp && maxp)) { return; }
    const avg = vector.multiply(vector.add(minp, maxp), 0.5);
    const station = Station.get(avg)
    if (station == undefined) {
        return;
    }
    const [vm, emin, emax] = core.get_mapgen_object("voxelmanip")
    // core.log("none", `generating minp=${minp.x},${minp.y},${minp.z}  maxp=${maxp.x},${maxp.y},${maxp.z}  emin=${emin.x},${emin.y},${emin.z}  emax=${emax.x},${emax.y},${emax.z}`)
    station.draw_all(new Drawer(vm), emin, emax)
    vm.set_lighting({ day: 0, night: 0 })
    vm.calc_lighting()
    vm.update_liquids()
    vm.write_to_map()
}

core.register_on_generated(generate)