--[[
    Convenience tools for Areas protection mod
    Copyright (C) 2025 rangeTonic

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, see <https://www.gnu.org/licenses/>.
--]]

local AS = core.get_translator("areas")
local S = core.get_translator(core.get_current_modname())
local FS = function(...) return core.formspec_escape(S(...)) end

local PROTECT_FORMSPEC_NAME = "areas_helper_tools:protect"
local PROTECT_BTN = "protect"
local PROTECT_NAME_FIELD = "newname"
local ADD_BTN = "add"
local ADD_PID_FIELD = "addpid"
local ADD_OWNER_FIELD = "addowner"
local ADD_NAME_FIELD = "addname"

local SELECT_FORMSPEC_NAME = "areas_helper_tools:select"
local SELECT_AREAS_LIST = "areas"
local SELECT_OK_BTN = "ok"
local SELECT_RM_BTN = "remove"

local CONFIRMRM_FORMSPEC_NAME = "areas_helper_tools:confirmrm"
local CONFIRMRM_YES_BTN = "yes"
local CONFIRMRM_NO_BTN = "no"

---Per-player timestamps of last area lookup.
---Stored for rate-limiting purposes.
---@type table<string, number>
local last_lookup = {}

---Per-player lists of found areas.
---@type table<string, integer[]>
local lookup_results = {}

---Per-player selected area indexes from `lookup_results`.
---@type table<string, integer>
local last_selected_result = {}

---@param user mt.PlayerObjectRef
---@param pointed_thing mt.PointedThing
---@return mt.Vector?
local function get_pointed_pos(user, pointed_thing)
	if pointed_thing.type == "node" then
		return pointed_thing.under
	elseif pointed_thing.type == "nothing" then
		local pos = user:get_pos()
		local dir = user:get_look_dir()
		-- TODO: get pointing range of player?
		return vector.round({
			x = pos.x + (dir.x * 4),
			y = pos.y + 1 + (dir.y * 4),
			z = pos.z + (dir.z * 4)
		})
	end
	return nil
end

---@param areaid integer
---@return string
local function get_confirmrm_formspec(areaid)
	local formspec = {
		"formspec_version[6]",
		"size[9.75,3]",
		"style[", CONFIRMRM_YES_BTN, ";bgcolor=red]",
		"label[0.375,0.5;", S("Remove this area and its sub areas?"), "]",
		"label[0.375,1;", core.formspec_escape(areas:toString(areaid)), "]",
		"button_exit[0.375,1.75;3,0.75;", CONFIRMRM_YES_BTN, ";", S("Yes"), "]",
		"button_exit[6.375,1.75;3,0.75;", CONFIRMRM_NO_BTN, ";", S("No"), "]",
		"set_focus[", CONFIRMRM_NO_BTN, ";true]",
	}
	return table.concat(formspec, "")
end

---@param pos1 mt.Vector
---@param pos2 mt.Vector
---@return string
local function get_protect_formspec(pos1, pos2)
	local formspec = {
		"formspec_version[6]",
		"size[8.75,8.25]",
		"position[0.5,0.5]",
		"label[0.375,0.5;", FS("Position @1: @2", "1", core.pos_to_string(pos1)), "]",
		"label[0.375,1;", FS("Position @1: @2", "2", core.pos_to_string(pos2)), "]",
		"box[0.375,1.45;8,0.05;#AAAAAA]",
		"label[0.375,2;", FS("Protect an area for yourself"), "]",
		"field[0.375,2.75;5.75,0.75;", PROTECT_NAME_FIELD, ";", FS("Area name"), ";]",
		"button_exit[6.375,2.75;2,0.75;", PROTECT_BTN, ";", FS("Protect"), "]",
		"box[0.375,3.95;8,0.05;#AAAAAA]",
		"label[0.375,4.5;", FS("Create a sub-area for another player\n(see /help add_owner)"), "]",
		"field[0.375,5.75;2.5,0.75;", ADD_PID_FIELD, ";", FS("Parent ID"), ";]",
		"field[3.125,5.75;3,0.75;", ADD_OWNER_FIELD, ";", FS("Player name"), ";]",
		"field[0.375,7;5.75,0.75;", ADD_NAME_FIELD, ";", FS("Area name"), ";]",
		"button_exit[6.375,7;2,0.75;", ADD_BTN, ";", FS("Add"), "]",
	}
	return table.concat(formspec, "")
end

---@param areaids integer[]
---@return string
local function get_select_formspec(areaids)
	local rows = {}
	for _, i in ipairs(areaids) do
		local arname = areas:toString(i)
		if arname:sub(1, 1) == "#" then
			arname = "##" .. arname
		end
		table.insert(rows, core.formspec_escape(arname))
	end

	local formspec = {
		"formspec_version[6]",
		"size[9.75,5.5]",
		"style[", SELECT_RM_BTN, ";bgcolor=red]",
		"label[0.375,0.5;", FS("Areas found: @1", #rows), "]",
		"textlist[0.375,1;9,3;", SELECT_AREAS_LIST, ";", table.concat(rows, ","), ";1;false]",
		"button[0.375,4.25;5.75,0.75;", SELECT_OK_BTN, ";", FS("Select"), "]",
		"button[6.375,4.25;3,0.75;", SELECT_RM_BTN, ";", FS("Remove"), "]",
	}
	return table.concat(formspec, "")
end

---@param user mt.PlayerObjectRef
---@param pointed_thing mt.PointedThing
---@param n 1|2
---@return nil
local function area_select(user, pointed_thing, n)
	local pname = user:get_player_name()
	local pos = get_pointed_pos(user, pointed_thing)
	if not pos then return end

	if n == 2 then
		areas:setPos2(pname, pos)
	else
		areas:setPos1(pname, pos)
	end
	core.chat_send_player(pname, AS("Area position @1 set to @2", n,
			core.pos_to_string(pos)))
end

---@param _ mt.ItemStack
---@param user mt.PlayerObjectRef
---@param pointed_thing mt.PointedThing
---@return nil
local function on_use(_, user, pointed_thing)
	if not (user and user:is_player()) then return end

	if not user:get_player_control().sneak then
		area_select(user, pointed_thing, 1)
		return
	end

	local pname = user:get_player_name()

	-- Only 1 area lookup per second
	local now = core.get_us_time()
	if now - (last_lookup[pname] or 0) >= 1000000 then
		last_lookup[pname] = now
	else
		return
	end

	local pos = get_pointed_pos(user, pointed_thing)
	if not pos then return end

	local foundIDs = {}
	for id, _ in pairs(areas:getAreasAtPos(pos)) do
		if areas:isAreaOwner(id, pname) then
			table.insert(foundIDs, id)
		end
	end

	if #foundIDs == 0 then
		core.chat_send_player(pname, S("No areas found."))
	else
		lookup_results[pname] = foundIDs
		last_selected_result[pname] = 1
		core.show_formspec(pname, SELECT_FORMSPEC_NAME,
				get_select_formspec(lookup_results[pname]))
	end
end

---@param _ mt.ItemStack
---@param user mt.PlayerObjectRef
---@param pointed_thing mt.PointedThing
---@return nil
local function on_place(_, user, pointed_thing)
	if not (user and user:is_player()) then return end

	if not user:get_player_control().sneak then
		area_select(user, pointed_thing, 2)
		return
	end

	local pname = user:get_player_name()
	local privs = core.get_player_privs(pname)
	-- TODO: consider adding extra priv
	if not privs[areas.config.self_protection_privilege] then
		core.chat_send_player(pname, S("You are not allowed to protect areas."))
		return
	end

	local pos1, pos2 = areas.pos1[pname], areas.pos2[pname]
	if not (pos1 and pos2) then
		core.chat_send_player(pname, AS("You need to select an area first."))
		return
	end

	core.show_formspec(pname, PROTECT_FORMSPEC_NAME,
			get_protect_formspec(pos1, pos2))
end

local mcl = Areas_protect_tool.mcl_available
local mtg = Areas_protect_tool.mtg_available
core.register_tool("areas_helper_tools:selector", {
	description = S("Area select-n-protect tool\nSneak+LMB: select existing.\nSneak+RMB: protect selection."),
	short_description = S("Area select-n-protect tool"),
	inventory_image = "areas_helper_tools_selector.png",
	wield_image = mcl and "areas_helper_tools_selector.png^[transform3" or nil,
	groups = mtg and {wieldview_transform = 3} or nil,
	pointabilities = {objects = {
		["areas:pos1"] = false,
		["areas:pos2"] = false
	}},
	on_place = on_place,
	on_use = on_use,
	on_secondary_use = on_place,
	_mcl_toollike_wield = mcl and true or nil
})

core.register_on_player_receive_fields(function(player, formname, fields)
	if formname == SELECT_FORMSPEC_NAME then
		local areaid
		local pname = player:get_player_name()
		if not lookup_results[pname] then return true end

		if fields[SELECT_OK_BTN] then
			areaid = lookup_results[pname][last_selected_result[pname] or 1]
		elseif fields[SELECT_RM_BTN] then
			areaid = lookup_results[pname][last_selected_result[pname] or 1]
			if areaid then
				core.show_formspec(pname, CONFIRMRM_FORMSPEC_NAME,
						get_confirmrm_formspec(areaid))
			end
			return true
		elseif fields[SELECT_AREAS_LIST] then
			local evt = core.explode_textlist_event(fields[SELECT_AREAS_LIST])
			if evt.type == "DCL" then
				areaid = lookup_results[pname][evt.index]
			else
				if evt.type == "CHG" then
					last_selected_result[pname] = evt.index
				end
				return true
			end
		else
			return true
		end
		if areaid then
			Areas_protect_tool.select_area(pname, areaid)
			core.close_formspec(pname, SELECT_FORMSPEC_NAME)
		end
		return true
	elseif formname == PROTECT_FORMSPEC_NAME then
		if fields[PROTECT_BTN] or fields.key_enter_field == PROTECT_NAME_FIELD then
			local arname = (fields[PROTECT_NAME_FIELD] or ""):trim()
			if #arname == 0 then arname = "Unnamed area" end
			Areas_protect_tool.protect(player:get_player_name(), arname)
		elseif fields[ADD_BTN] or (fields.key_enter == "true" and (
				fields.key_enter_field == ADD_PID_FIELD or
				fields.key_enter_field == ADD_OWNER_FIELD or
				fields.key_enter_field == ADD_NAME_FIELD)) then

			local arname = (fields[ADD_NAME_FIELD] or ""):trim()
			if #arname == 0 then arname = "Unnamed area" end
			local pname = player:get_player_name()

			local pid = (fields[ADD_PID_FIELD] or ""):match("^%s*(%d+)%s*$")
			if not pid then
				core.chat_send_player(pname, core.colorize("#b57614",
						S("Missing or malformed parent ID.")))
				return true
			end

			local owner = (fields[ADD_OWNER_FIELD] or ""):match("^%s*([0-9A-Za-z_-]+)%s*$")
			if not owner then
				core.chat_send_player(pname, core.colorize("#b57614",
						S("Missing or malformed player name.")))
				return true
			end

			Areas_protect_tool.add_owner(pname, tonumber(pid), owner, arname)
		end
		return true
	elseif formname == CONFIRMRM_FORMSPEC_NAME then
		if fields[CONFIRMRM_YES_BTN] then
			local pname = player:get_player_name()
			if lookup_results[pname] then
				local areaid = lookup_results[pname][last_selected_result[pname] or 1]
				if areaid then
					Areas_protect_tool.remove(pname, areaid, true)
				end
			end
		end
		return true
	end
end)

core.register_on_leaveplayer(function(player)
	local pname = player:get_player_name()
	last_lookup[pname] = nil
	lookup_results[pname] = nil
	last_selected_result[pname] = nil
end)
