-- copyright (c) 2024 rstcxk
-- 
-- this program is free software: you can redistribute it and/or modify it under the terms of
-- the gnu affero general public license as published by the free software foundation, either version 3 of the license, or (at your option) any later version.
-- 
-- this program is distributed in the hope that it will be useful, but without any warranty;
-- without even the implied warranty of merchantability or fitness for a particular purpose. see the gnu affero general public license for more details.
-- 
-- you should have received a copy of the gnu affero general public license along with this program. if not, see <https://www.gnu.org/licenses/>. 

builders_wand = {}

local consts = {
	recent_history = 80
}

-- player data is a dictionary of player names (keys) and entries
-- each entry has the following members:
-- selected_item (node name)
-- selected_group (name of group)
-- replace_mode (bool)
-- recent_items
-- recent_groups
-- custom_groups (dict of groups) - for optimization reasons, they're excluded from the searchbar
-- extruder_propagation_mode - either "Neighbours" or "All adjescent"
-- extruder_mode = either of: "Smart", "Dumb"
builders_wand.player_data = {}

local storage = core.get_mod_storage()

function builders_wand.initialize_player_data(player)
	if not builders_wand.get_data_from_player(player) then
		builders_wand.player_data[player:get_player_name()] = {}
		local data = builders_wand.player_data[player:get_player_name()]
		data.recent_items = {}
		data.recent_groups = {}
		data.recent_groups = {}
		data.custom_groups = {}
	end
end

function builders_wand.serialize_data()
	storage:set_string("player_data", core.serialize(builders_wand.player_data))
end

function builders_wand.deserialize_data()
	local storage_value = storage:get_string("player_data")
	builders_wand.player_data = storage_value ~= "" and core.deserialize(storage_value) or builders_wand.player_data
end

function builders_wand.get_data_from_name(player_name)
	return builders_wand.player_data[player_name]
end

function builders_wand.get_data_from_player(player)
	return builders_wand.player_data[player:get_player_name()]
end

function builders_wand.select_item(data, new_node)
	data.selected_item = new_node

	-- if there are any repeats, don't duplicate, but move to the top of the queue
	for i = #data.recent_items, 1, -1 do
		if(data.recent_items[i] == new_node) then
			table.remove(data.recent_items, i)
			break
		end
	end

	if #data.recent_items >= consts.recent_history then
		data.recent_items[#data.recent_items] = nil
	end

	table.insert(data.recent_items, 1, new_node)
end

function builders_wand.select_group(data, new_group)
	data.selected_group = new_group

	-- if there are any repeats, don't duplicate, but move to the top of the queue
	for i = #data.recent_groups, 1, -1 do
		if(data.recent_groups[i] == new_group) then
			table.remove(data.recent_groups, i)
			break
		end
	end

	if #data.recent_groups >= consts.recent_history then
		data.recent_groups[#data.recent_groups] = nil
	end

	table.insert(data.recent_groups, 1, new_group)
end

builders_wand.deserialize_data()

core.register_on_joinplayer(function(player)
	builders_wand.initialize_player_data(player)
end)

core.register_on_shutdown(function()
	builders_wand.serialize_data()
end)

dofile(core.get_modpath("builders_wand") .. "/api.lua")
dofile(core.get_modpath("builders_wand") .. "/place.lua")
dofile(core.get_modpath("builders_wand") .. "/cmd.lua")
dofile(core.get_modpath("builders_wand") .. "/extruder.lua")
dofile(core.get_modpath("builders_wand") .. "/tools.lua")
dofile(core.get_modpath("builders_wand") .. "/marker.lua")
builders_wand.hud = dofile(core.get_modpath("builders_wand") .. "/hud.lua")
