-- {{{ search offsets
local search_offsets = {
	[core.hash_node_position(vector.new(1, 0, 0))] = {
			vector.new(0, 1,  0),
			vector.new(0, -1, 0),
			vector.new(0, 0,  1),
			vector.new(0, 0,  -1),
	},
	[core.hash_node_position(vector.new(-1, 0, 0))] = {
			vector.new(0, 1,  0),
			vector.new(0, -1, 0),
			vector.new(0, 0,  1),
			vector.new(0, 0,  -1),
	},
	[core.hash_node_position(vector.new(0, 1, 0))] = {
		vector.new(1,  0, 0),
		vector.new(-1, 0, 0),
		vector.new(0,  0, 1),
		vector.new(0,  0, -1),
	},
	[core.hash_node_position(vector.new(0, -1, 0))] = {
		vector.new(1,  0, 0),
		vector.new(-1, 0, 0),
		vector.new(0,  0, 1),
		vector.new(0,  0, -1),
	},
	[core.hash_node_position(vector.new(0, 0, 1))] = {
		vector.new(1,  0, 0),
		vector.new(-1, 0, 0),
		vector.new(0,  1, 0),
		vector.new(0, -1, 0),
	},
	[core.hash_node_position(vector.new(0, 0, -1))] = {
		vector.new(1,  0, 0),
		vector.new(-1, 0, 0),
		vector.new(0,  1, 0),
		vector.new(0, -1, 0),
	}
}

local diagonal_search_offsets = {
	[core.hash_node_position(vector.new(1, 0, 0))] = {
			vector.new(0, 1,  0),
			vector.new(0, -1, 0),
			vector.new(0, 0,  1),
			vector.new(0, 0,  -1),

			vector.new(0, 1,  -1),
			vector.new(0, 1,  1),
			vector.new(0, -1,  -1),
			vector.new(0, -1,  1),
	},
	[core.hash_node_position(vector.new(-1, 0, 0))] = {
			vector.new(0, 1,  0),
			vector.new(0, -1, 0),
			vector.new(0, 0,  1),
			vector.new(0, 0,  -1),

			vector.new(0, 1,  -1),
			vector.new(0, 1,  1),
			vector.new(0, -1,  -1),
			vector.new(0, -1,  1),
	},
	[core.hash_node_position(vector.new(0, 1, 0))] = {
		vector.new(1,  0, 0),
		vector.new(-1, 0, 0),
		vector.new(0,  0, 1),
		vector.new(0,  0, -1),

		vector.new(1,  0, -1),
		vector.new(1,  0, 1),
		vector.new(-1,  0, -1),
		vector.new(-1,  0, 1),
	},
	[core.hash_node_position(vector.new(0, -1, 0))] = {
		vector.new(1,  0, 0),
		vector.new(-1, 0, 0),
		vector.new(0,  0, 1),
		vector.new(0,  0, -1),

		vector.new(1,  0, -1),
		vector.new(1,  0, 1),
		vector.new(-1,  0, -1),
		vector.new(-1,  0, 1),
	},
	[core.hash_node_position(vector.new(0, 0, 1))] = {
		vector.new(1,  0, 0),
		vector.new(-1, 0, 0),
		vector.new(0,  1, 0),
		vector.new(0, -1, 0),

		vector.new(1, -1, 0),
		vector.new(1, 1, 0),
		vector.new(-1, -1, 0),
		vector.new(-1, 1, 0),
	},
	[core.hash_node_position(vector.new(0, 0, -1))] = {
		vector.new(1,  0, 0),
		vector.new(-1, 0, 0),
		vector.new(0,  1, 0),
		vector.new(0, -1, 0),

		vector.new(1, -1, 0),
		vector.new(1, 1, 0),
		vector.new(-1, -1, 0),
		vector.new(-1, 1, 0),
	}
}
-- }}}

local extrusion_qualifiers = {
	["Smart"] = {
		qualifier = function(node)
			local ndef = core.registered_nodes[node.name]
			return node.name ~= "air" and not ndef.buildable_to and ndef.walkable
		end,
		qualifier_cover = function(node)
			if node.name == "air" then
				return "pass"
			else
				local ndef = core.registered_nodes[node.name]

				if ndef.buildable_to or not ndef.walkable then
					return "pass_log"
				end
			end
			return "ignore"
		end
	},
	["Dumb"] = {
		qualifier = function(node)
			return node.name ~= "air"
		end,
		qualifier_cover = function(node)
			return node.name == "air" and "pass" or "ignore"
		end
	}
}

-- qualifier_cover returns either:
-- "pass" if the face should count
-- "ignore" if the face shouldn't count
-- "pass_log" if the face should count but the node be saved
--
-- qualifier returns simply true or false whether the node should propagate the face
local function get_faces(pos, dir, mode, qualifier, qualifier_cover)
	local faces = {}
	local already_added = {}
	local face_aux_log = {} -- the PAIN of naming variables

	local function attempt_extrusion(pos)
		local hash = core.hash_node_position(pos)
		local front_node = core.get_node(pos:add(dir))
		local node = core.get_node(pos)
		local qualifier_cover_result = qualifier_cover(front_node)
		local qualifier_result = qualifier(node)
		if not already_added[hash]
				and qualifier_result
				and qualifier_cover_result ~= "ignore" then

			if qualifier_cover_result == "pass_log" then
				face_aux_log[pos] = front_node
			end

			table.insert(faces, pos)
			already_added[hash] = true
		end
	end

	local offsets
	if mode == "Neighbours" then
		offsets = search_offsets[core.hash_node_position(dir)]
	elseif mode == "All adjecent" then
		offsets = diagonal_search_offsets[core.hash_node_position(dir)]
	else
		error(string.format("invalid mode for `get_faces`: %s", mode))
	end

	attempt_extrusion(pos)

	local i = 0
	local frustration = 25000
	while i ~= #faces and i < frustration do
		i = i + 1
		local pos = faces[i]
		for _, search_offset in pairs(offsets) do
			attempt_extrusion(vector.add(pos, search_offset))
		end
	end

	if i >= frustration then
		core.log("warning", "tried to extend an unreasonable amount of nodes. Dropping the attempt")
		return {}, {}
	end

	return faces, face_aux_log
end

function builders_wand.extrude(pos, dir, propagation_mode, mode)
	local faces, push_nodes = get_faces(
		pos,
		dir,
		propagation_mode,
		extrusion_qualifiers[mode].qualifier,
		extrusion_qualifiers[mode].qualifier_cover
	)

	for _, v in pairs(faces) do
		local node = core.get_node(v)
		core.set_node(vector.add(v, dir), node)
	end

	for pos, node in pairs(push_nodes) do
		core.set_node(pos + dir + dir, node)
	end
end

function builders_wand.indent(pos, dir, propagation_mode, mode)
	-- local faces, drag_along = get_faces(pos, dir), {name = "air"}
	local faces, drag_along = get_faces(
		pos,
		dir,
		propagation_mode,
		extrusion_qualifiers[mode].qualifier,
		extrusion_qualifiers[mode].qualifier_cover
	)

	core.bulk_set_node(faces, {name = "air"})

	for pos, node in pairs(drag_along) do
		core.remove_node(pos + dir)
		if core.get_node(pos - dir).name ~= "air" then
			-- prevent leaving floating torches for example
			core.set_node(pos, node)
		end
	end
end

function builders_wand.paint_face(pos, dir, to_node)
	local faces, _ = get_faces(
		pos,
		dir,
		"All adjecent",
		extrusion_qualifiers["Smart"].qualifier,
		extrusion_qualifiers["Smart"].qualifier_cover
	)

	core.bulk_set_node(faces, to_node)
end

function builders_wand.scrape_face(pos, dir)
	local _, to_be_removed = get_faces(
		pos,
		dir,
		"All adjecent",
		extrusion_qualifiers["Smart"].qualifier,
		extrusion_qualifiers["Smart"].qualifier_cover
	)

	for pos, node in pairs(to_be_removed) do 
		core.remove_node(pos + dir)
	end
end
