-- copyright (c) 2024 rstcxk
-- 
-- this program is free software: you can redistribute it and/or modify it under the terms of
-- the gnu affero general public license as published by the free software foundation, either version 3 of the license, or (at your option) any later version.
-- 
-- this program is distributed in the hope that it will be useful, but without any warranty;
-- without even the implied warranty of merchantability or fitness for a particular purpose. see the gnu affero general public license for more details.
-- 
-- you should have received a copy of the gnu affero general public license along with this program. if not, see <https://www.gnu.org/licenses/>. 

-- table whose keys are group names, and values are a list of node names that belong to the group
builders_wand.node_groups = {}

-- a dictionary where the key is the group name, and the value is the SORT NAME
-- sort names are names used in sorting, so related groups can show near eachother
-- This would be cleaner if i could store this in group's definition, but that would require a lot of refactoring
builders_wand.sort_names = {}

-- table whose keys are node names, and values are a list of groups names that the node belongs to
builders_wand.node_belongings = {}

local modpath = core.get_modpath("builders_wand")

function builders_wand.register_group(def, group_content)
	builders_wand.node_groups[def.group_name] = group_content
	builders_wand.sort_names[def.group_name] = def.sort_name

	for _, v in pairs(group_content) do
		if(builders_wand.node_belongings[v]) then
			table.insert(builders_wand.node_belongings[v], def.group_name)
		else
			builders_wand.node_belongings[v] = {def.group_name}
		end
	end
end

function builders_wand.get_groups(player)
	local output = {}
	local data = builders_wand.get_data_from_player(player)

	-- This loop comes first, so it precedes all the other ones
	for v, _ in pairs(data.custom_groups) do
		table.insert(output, v)
	end

	local tmp = {}
	for v, _ in pairs(builders_wand.node_groups) do
		table.insert(tmp, v)
	end

	table.sort(tmp, function(group1, group2)
		-- THOS IS SO RETARDED
		local sort_name_1 = builders_wand.sort_names[group1]
		local sort_name_2 = builders_wand.sort_names[group2]
		if sort_name_1 and not sort_name_2 then
			return true
		elseif not sort_name_1 and sort_name_2 then
			return false
		elseif sort_name_1 and sort_name_2 then
			if sort_name_1 == sort_name_2 then
				return group1 > group2
			end
			return sort_name_1 > sort_name_2
		else
			return group1 > group2
		end
	end)

	for _, v in pairs(tmp) do
		table.insert(output, v)
	end

	return output
end

-- meant to be overriden by game specific logic
function builders_wand.is_player_inventory_list_relevant(listname)
	return true
end

function builders_wand.get_groups_items(groupname, player)
	if builders_wand.node_groups[groupname] then
		return builders_wand.node_groups[groupname]
	end

	local data = builders_wand.get_data_from_player(player)
	return data.custom_groups[groupname]
end

function builders_wand.get_group_type(groupname, player)
	if builders_wand.node_groups[groupname] then
		return "global"
	end

	local data = builders_wand.get_data_from_player(player)
	return data.custom_groups[groupname] and "custom" or "non existent"

end

function builders_wand.get_color_for_group_type(type)
	if type == "global" then
		return "#FFFFFF"
	elseif type == "custom" then
		return "#00FF00"
	else
		return "#FF0000"
	end
end

function builders_wand.get_formatted_group_name(groupname, player)
	return core.colorize(builders_wand.get_color_for_group_type(builders_wand.get_group_type(groupname, player)), groupname)

end

-- TODO: make sure no duplicates are not added
function builders_wand.custom_group_add_from_inv(groupname, player)
	local group = builders_wand.get_data_from_player(player).custom_groups[groupname]

	local group_set = {}

	for _, v in pairs(group) do
		group_set[v] = true
	end

	local player_inv = player:get_inventory()

	for listname, list in pairs(player_inv:get_lists()) do
		if builders_wand.is_player_inventory_list_relevant(listname) then
			for _, item in pairs(list) do
				local item_name = item:get_name()
				if not group_set[item_name] and core.registered_nodes[item_name] then
					table.insert(group, item_name)
					group_set[item_name] = true
				end
			end
		end
	end
end

function builders_wand.create_custom_group(groupname, player)
	builders_wand.get_data_from_player(player).custom_groups[groupname] = {}
end

-- TODO: make sure no duplicates are not added
function builders_wand.add_item_to_custom_group(groupname, player, item)
	local group = builders_wand.get_data_from_player(player).custom_groups[groupname]
	for _, v in pairs(group) do
		if item == v then
			return
		end
	end
	table.insert(group, item)
end

function builders_wand.remove_item_from_custom_group(groupname, player, item)
	local idx = -1
	local group = builders_wand.get_data_from_player(player).custom_groups[groupname]
	for i, v in pairs(group) do
		if v == item then
			idx = i
			break
		end
	end
	table.remove(group, idx)
end

function builders_wand.delete_custom_group(groupname, player)
	if builders_wand.get_group_type(groupname, player) ~= "custom" then return end

	local data = builders_wand.get_data_from_player(player)
	data.custom_groups[groupname] = nil

	if data.selected_group == groupname then
		data.selected_group = nil
	end

	local delete_idx
	for i, v  in pairs(data.recent_groups) do
		if v == groupname then
			delete_idx = i
			break
		end
	end

	if delete_idx then
		table.remove(data.recent_groups, delete_idx)
	end
end

function builders_wand.clear_history(player)
	local data = builders_wand.get_data_from_player(player)

	data.recent_groups = {}
	data.recent_items = {}
end

function builders_wand.add_node_to_group(group_name, node_name)
	if not builders_wand.node_groups[group_name] then
		return false
	end
	table.insert(builders_wand.node_groups[group_name], node_name)

	if(builders_wand.node_belongings[node_name]) then
		table.insert(builders_wand.node_belongings[node_name], group_name)
	else
		builders_wand.node_belongings[node_name] = {group_name}
	end

	return true
end

if core.global_exists("mcl_core") then
	dofile(modpath .. "/defs/mcl_defs.lua")
end

if core.get_modpath("default") then
	dofile(modpath .. "/defs/minetest_defs.lua")
end
