local gui_helpers = {}
local gui = flow.widgets

local columns_per_row = 10
local dialog_width = 17

local function gui_create_scrollarea(items, accessor, labeller, callback)
	local output_list = {}
	local head
	for i, v in pairs(items) do
		if (i - 1) % columns_per_row == 0 then
			if head then
			table.insert(head, gui.Spacer{})
			table.insert(output_list, head)
			end
			head = gui.Hbox{}
		end

		table.insert(head, gui.ItemImageButton{
			item_name = accessor(v),
			label = labeller and labeller(v),
			w = 1.5,
			h = 1.5,
			on_event = function(player, ctx)
				callback(player, ctx, v)
			end})
	end

	if head and #head ~= 0 then
		table.insert(output_list, head)
	end

	return output_list
end

function gui_helpers.get_all_groups_matching_query(query)
	query = string.lower(query)
	local group_matches = {}
	for group_name, _ in pairs(builders_wand.node_groups) do
		if string.find(group_name, query, nil, true) then
			group_matches[group_name] = group_matches[group_name] and group_matches[group_name] + 10 or 1
		end
	end

	for node_name, group_names in pairs(builders_wand.node_belongings) do
		if string.find(node_name, query, nil, true)
			or (core.registered_nodes[node_name].description and string.find(string.lower(core.registered_nodes[node_name].description), query, nil, true)) then
			for _, group_name in pairs(group_names) do
				group_matches[group_name] = group_matches[group_name] and group_matches[group_name] + 1 or 1
			end
		end
	end

	local output_list = {}

	for group_name, _ in pairs(group_matches) do
		table.insert(output_list, group_name)
	end

	table.sort(output_list, function(lhs, rhs) return group_matches[lhs] > group_matches[rhs] end)

	return output_list
end

function gui_helpers.create_group_selection_gui(player)
	local my_gui

	local function get_scrollarea(ctx, my_gui)
		local iterable = ctx.matches or builders_wand.get_groups(player)
		return gui_create_scrollarea(iterable,
		function(v)
			return builders_wand.get_groups_items(v, player)[1]
		end,
		function(v)
			return builders_wand.get_formatted_group_name(v, player)
		end,
		function(player, ctx, v)
			local builder_data = builders_wand.get_data_from_player(player)
			builders_wand.select_group(builder_data, v)
			my_gui:close(player)
			builders_wand.create_item_selection_gui(player):show(player)
		end)
	end

	my_gui = flow.make_gui(function(player, ctx)
		return gui.VBox
		{
			gui.HBox {
				gui.Label{label = "search groups"},
				gui.Field{
					name = "searchbar",
					margin = 2,
					expand = true,
					on_event = function(player, ctx)
						if ctx.form.searchbar == "" then
							ctx.matches = nil
						else
							ctx.matches = gui_helpers.get_all_groups_matching_query(ctx.form.searchbar)
						end
						return true
					end
				},
			},
			gui.ScrollableVBox {
				name = "scroll",
				h = 7,
				w = dialog_width,

				unpack(get_scrollarea(ctx, my_gui))
			},
		}
	end)

	return my_gui
end

function builders_wand.create_item_selection_gui(player)
	local builder_data = builders_wand.get_data_from_player(player)
	local group_name = builder_data.selected_group
	local my_gui

	local function get_scrollarea(groupname)
		return gui_create_scrollarea(builders_wand.get_groups_items(groupname, player),
		function(v)
			return v
		end,
		nil,
		function(player, ctx, v)
			builders_wand.select_item(builder_data, v)
		end)
	end

	my_gui = flow.make_gui(function(player, ctx)
		return gui.VBox
		{
			gui.HBox {
				gui.Label{label = "builder's wand"},
				gui.Checkbox{
					name = "replace_mode_checkbox",
					label = "replace mode",
					selected = builder_data.replace_mode,
					on_event = function(player, ctx)
						builder_data.replace_mode = ctx.form.replace_mode_checkbox
					end
				},
				gui.Spacer{},
				gui.Button{
					label = "search groups",
					align_h = "right",
					on_event = function(player, ctx)
						my_gui:close(player)
						gui_helpers.create_group_selection_gui(player):show(player)
					end
				}
			},
			gui.ScrollableVBox {
				name = "scroll",
				h = 7,
				w = dialog_width,

				unpack(get_scrollarea(group_name))
			},
		}
	end)

	return my_gui
end

function gui_helpers.create_recent_items_selection_gui(player)
	local builder_data = builders_wand.get_data_from_player(player)
	local my_gui

	local function get_recent_items()
		return gui_create_scrollarea(builder_data.recent_items,
		function(v)
			 return v
		end,
		nil,
		function(player, ctx, v)
			builders_wand.select_item(builder_data, v)
		end)
	end

	local function get_groups_items()
		return gui_create_scrollarea(builder_data.recent_groups,
		function(v)
			local group_items = builders_wand.get_groups_items(v, player)
			return group_items and group_items[1] or ""
		end,
		function(v)
			return v
		end,
		function(player, ctx, v)
			builders_wand.select_group(builder_data, v)
			my_gui:close(player)
			builders_wand.create_item_selection_gui(player):show(player)
		end)
	end

	my_gui = flow.make_gui(function(player, ctx)
		return gui.VBox
		{
			gui.HBox {
				gui.Label{label = "Recent items"},
				gui.Checkbox{
					name = "replace_mode_checkbox",
					label = "replace mode",
					selected = builder_data.replace_mode,
					on_event = function(player, ctx)
						builder_data.replace_mode = ctx.form.replace_mode_checkbox
					end
				},
				gui.Spacer{},
				gui.Button{
					label = "Clear history",
					selected = builder_data.replace_mode,
					on_event = function(player, ctx)
						builders_wand.clear_history(player)
						return true
					end
				},
			},
			gui.ScrollableVBox {
				name = "recent_items_scroll",
				h = 3,
				w = dialog_width,

				unpack(get_recent_items())
			},
			gui.Label{label = "Recent groups"},
			gui.ScrollableVBox {
				name = "recent_groups_scroll",
				h = 3,
				w = dialog_width,

				unpack(get_groups_items())
			},
		}
	end)

	return my_gui
end

function gui_helpers.create_options_gui(player)
	local builder_data = builders_wand.get_data_from_player(player)

	local my_gui = flow.make_gui(function(player, ctx)
		return gui.VBox
		{
			gui.HBox {
				gui.Label{label = "Options"},
			},
			gui.HBox {
				gui.Checkbox{
					name = "show_builder_hud",
					label = "show builder HUD",
					selected = builder_data.show_hud,
					on_event = function(player, ctx)
						builder_data.show_hud = ctx.form.show_builder_hud
					end
				},
			},
		}
	end)

	return my_gui
end

-- callback is a function with parameters: player, ctx, v (selected item name)
function gui_helpers.create_item_selection_gui_for_group(player, callback)
	local builder_data = builders_wand.get_data_from_player(player)
	local my_gui

	local function get_scrollarea(groupname)
		return gui_create_scrollarea(builders_wand.get_groups_items(groupname, player),
		function(v)
			return v
		end,
		nil,
		callback)
	end

	my_gui = flow.make_gui(function(player, ctx)
		return gui.VBox
		{
			gui.HBox {
				gui.Label{label = "Select item"},
			},
			gui.ScrollableVBox {
				name = "scroll",
				h = 7,
				w = dialog_width,

				unpack(get_scrollarea(builder_data.selected_group))
			},
		}
	end)

	return my_gui
end

function gui_helpers.create_group_manager_gui(player)
	local builder_data = builders_wand.get_data_from_player(player)

	local edit_section = {}
	local currently_selected_group_type = builders_wand.get_group_type(builder_data.selected_group, player)

	local function update(player)
		gui_helpers.create_group_manager_gui(player):show(player)
	end

	local function show_item_deletion_dialog(player, ctx)
		gui_helpers.create_item_selection_gui_for_group(player, function(player, ctx, v)
			builders_wand.remove_item_from_custom_group(builder_data.selected_group, player, v)
			show_item_deletion_dialog(player, ctx)
		end):show(player)
	end

	if currently_selected_group_type == "custom" then
		edit_section = {
			gui.HBox {
				gui.Button{
					label = "Add nodes from inventory",
					name = "add_nodes_from_inventory",
					on_event = function(player, ctx)
						builders_wand.custom_group_add_from_inv(builder_data.selected_group, player)
					end
				},
				gui.Button{
					label = "Delete item",
					on_event = show_item_deletion_dialog
				},
				gui.Button{
					label = "Delete group",
					on_event = function(player, ctx)
						builders_wand.delete_custom_group(builder_data.selected_group, player)
						update(player)
						-- return true
					end
				},
			},
		}
	else
		edit_section = {
			gui.HBox {
				gui.Label{
					label = "Only user defined groups can be edited",
				},
			},

		}
	end


	local my_gui = flow.make_gui(function(player, ctx)
		return gui.VBox
		{
			gui.HBox {
				gui.Label{label = "Group manager"},
				gui.Label{label = "currently selected group: " .. (builder_data.selected_group
					and builders_wand.get_formatted_group_name(builder_data.selected_group, player)
					or core.colorize("#FF0000", "None"))
				},
			},
			gui.HBox {
				gui.Field{
					label = "New group name: ",
					name = "new_group_name"
				},
				gui.Button{
					label = "Create",
					name = "create_new_group_button",
					on_event = function(player, ctx)
						if #ctx.form.new_group_name == 0 then
							return
						end

						builders_wand.create_custom_group(ctx.form.new_group_name, player)
						builders_wand.select_group(
							builder_data,
							ctx.form.new_group_name
						)
						update(player)
						-- return true
					end
				},
			},
			gui.HBox {
				gui.Label{
					label = "Manage current group",
				},
			},
			unpack(edit_section),
			gui.Button {
				label = "Pick group",
				on_event = function(player, ctx)
					local data = builders_wand.get_data_from_player(player)
					local groups = {}
					for groupname, _ in pairs(data.custom_groups) do
						table.insert(groups, groupname)
					end
					flow.make_gui(function(player, ctx)
						return gui.VBox {
							gui.Label {
								label = "Pick a group"
							},
							gui.ScrollableVBox {
								name = "scroll",
								h = 7,
								w = dialog_width,

								unpack(gui_create_scrollarea(groups,
								function(v)
									local group_items = builders_wand.get_groups_items(v, player)
									return group_items and group_items[1] or ""
								end,
								nil,
								function(player, ctx, v)
									local data = builders_wand.get_data_from_player(player)
									builders_wand.select_group(data, v)
									update(player)
								end))
							},
						}
					end):show(player)
				end
			}
		}
	end)

	return my_gui
end

function gui_helpers.create_extruder_gui(player)
	local builder_data = builders_wand.get_data_from_player(player)

	local my_gui = flow.make_gui(function(player, ctx)
		return gui.VBox
		{
			gui.HBox {
				gui.Label{label = "Extruder options"},
			},
			gui.HBox {
				gui.Label{
					label = "Mode",
					w = 2,
				},
				gui.Dropdown{
					selected_idx = builder_data.extruder_mode == "Dumb" and 2 or 1,
					w = 4,
					name = "mode",
					items = {"Smart", "Dumb"},
					on_event = function(player, ctx)
						builder_data.extruder_mode = ctx.form.mode
					end
				},
			},
			gui.HBox {
				gui.Label{
					label = "Propagation Mode",
					w = 2,
				},
				gui.Dropdown{
					selected_idx = builder_data.extruder_propagation_mode == "All adjecent" and 2 or 1,
					w = 4,
					name = "propagation_mode",
					items = {"Neighbours", "All adjecent"},
					on_event = function(player, ctx)
						builder_data.extruder_propagation_mode = ctx.form.propagation_mode
					end
				},
			},
		}
	end)

	return my_gui
end

function gui_helpers.create_fast_placer_gui(player)
	local builder_data = builders_wand.get_data_from_player(player)

	local my_gui = flow.make_gui(function(player, ctx)
		return gui.VBox
		{
			gui.HBox {
				gui.Label{label = "Fast placer options"},
			},
			gui.Checkbox{
				name = "replace_mode_checkbox",
				label = "replace mode",
				selected = builder_data.replace_mode,
				on_event = function(player, ctx)
					builder_data.replace_mode = ctx.form.replace_mode_checkbox
				end
			},
			gui.HBox {
				gui.Checkbox{
					name = "axis_aligner_checkbox",
					label = "Axis aligner",
					selected = builder_data.fast_placer_alignment_enabled,
					on_event = function(player, ctx)
						builder_data.fast_placer_alignment_enabled = ctx.form.axis_aligner_checkbox
					end
				},
			},
			gui.HBox {
				gui.Label{
					label = "Speed",
					w = 2,
				},
				gui.Field{
					name = "speed",
					margin = 2,
					expand = true,
					default = tostring(builder_data.fast_placer_time),
					on_event = function(player, ctx)
						local value = tonumber(ctx.form.speed)
						if value then
							builder_data.fast_placer_time = value
						end
					end
				},
			},
		}
	end)

	return my_gui
end

return gui_helpers
