-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_ProcessSubstitution
--	describes a subshell similar to bash's process substitution

local helpers = require("helpers")
local Shell
local ShellContext
local argument_terminators =
{
	[" "] = true,
	["\t"] = true,
	[";"] = true,
	["|"] = true,
}

local ProcessSubstitution =
{
	instance_of = "LanguageConstruct_ProcessSubstitution",
}

ProcessSubstitution.__index = ProcessSubstitution

function ProcessSubstitution:new(t)
	t = t or {}
	self = setmetatable(
	{
		subshell = t.subshell
	}, ProcessSubstitution)

	return self
end

function ProcessSubstitution:evaluate(context)
	-- {{{ type checking
	if context.instance_of ~= "ShellContext" then error("context should be an instance of ShellContext. instead of " .. context.instance_of) end
	-- }}}

	ShellContext = ShellContext or require("shell_context")
	local subshell_context = ShellContext:new()
	subshell_context.env:inherit(context.env)
	self.subshell:evaluate(subshell_context)

	return subshell_context.stdout
end

function ProcessSubstitution:parse(parser_context)
	-- {{{ type checking
	if parser_context.instance_of ~= "ParserContext" then error("parser_context should be an instance of ParserContext. instead of " .. parser_context.instance_of) end
	-- }}}
	
	local Shell = Shell or require("language_constructs.shell")

	local char = parser_context:consume()

	if char ~= "$" then
		error("expected a \"$\"")
	end

	char = parser_context:consume()

	if char ~= "(" then
		error("expected a \"(\"")
	end

	self.subshell = Shell:new()
	self.subshell:parse(parser_context, {")"}, helpers.concat_sets({[")"] = true}, argument_terminators))

end

return ProcessSubstitution
