-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- this file contains simple commands that dont depend on minetest namespace

minetest.register_privilege("export",
{
	description = "whether the player can modify lush's global environment",
	give_to_singleplayer = true
})

lush.Command:new(
{
	name = "echo",
	stdin_type = "string",
	callback = function(context, args, options)
		for i, v in pairs(args) do
			if type(v) == "table" and v.instance_of == "TypedList" then
				for _, v2 in v:iterator() do
					context.stdout:push(v2)
				end
			else
				context.stdout:push(v)
			end
		end
	end,
	privs = {},
	description = "prints arguments to STDOUT",
	params = "<arguments>"
})

lush.Command:new(
{
	name = "append",
	validate_arguments =
	{
		{
			name = "postfix",
			type = "string",
			single = true
		}
	},
	stdin_type = "string",
	callback = function(context, args, options)
		for i, v in pairs(context.stdin.container) do
			context.stdout:push(v .. args["postfix"])
		end
	end,
	privs = {},
	description = "appends a string to every line in STDIN",
	params = "<string>"
})

lush.Command:new(
{
	name = "lines",
	validate_arguments =
	{
		{
			name = "range",
			type = "range",
			single = true
		}
	},
	callback = function(context, args, options)
		for line_num, line in context.stdin:iterator() do
			if args["range"]:contains(line_num) then
				context.stdout:push(line)
			end
		end
	end,
	privs = {},
	description = "takes STDIN and prints only lines in the range",
	params = "<range>"
})

lush.Command:new(
{
	name = "wc",
	validate_arguments = {},
	callback = function(context, args, options)
		context.stdout:push(#context.stdin.container)
	end,
	privs = {},
	description = "counts the amount of lines in STDIN",
	params = ""
})

lush.Command:new(
{
	name = "prepend",
	validate_arguments =
	{
		{
			name = "prefix",
			type = "string",
			single = true
		}
	},
	stdin_type = "string",
	callback = function(context, args, options)
		for i, v in pairs(context.stdin.container) do
			context.stdout:push(args["prefix"] .. v)
		end
	end,
	privs = {},
	description = "prepends all lines from STDIN with the given string",
	params = "<string>"
})

lush.Command:new(
{
	name = "set",
	validate_arguments =
	{
		{
			name = "param",
			type = "string",
			single = true
		}
	},
	callback = function(context, args, options)
		context.env[args["param"]] = context.stdin:get(1)
	end,
	privs = {},
	description = "sets the given parameter to the value of STDIN",
	params = "<param>"
})

lush.Command:new(
{
	name = "split",
	validate_arguments =
	{
		{
			name = "seperator",
			type = "string",
			single = true
		}
	},
	stdin_type = "string",
	callback = function(context, args, options)
		for i, v in pairs(context.stdin.container) do
			context.stdout:concat(lush.helpers.string_split(v, args["seperator"]))
		end
	end,
	privs = {},
	description = "splits STDIN into lines",
	params = "<seperator>"
})

lush.Command:new(
{
	name = "match",
	validate_arguments =
	{
		{
			name = "pattern",
			type = "string",
			single = true
		}
	},
	stdin_type = "string",
	callback = function(context, args, options)
		for i, v in pairs(context.stdin.container) do
			if string.find(v, args["pattern"]) then
				context.stdout:push(v)
			end

		end
	end,
	privs = {},
	description = "prints only the lines from STDIN that match the pattern",
	params = "<pattern>"
})

lush.Command:new(
{
	name = "exit",
	validate_arguments =
	{
		{
			name = "exit_status",
			type = "number",
			single = true
		}
	},
	callback = function(context, args, options)
		if args["exit_status"] ~= 0 then
			error()
		end
	end,
	privs = {},
	description = "exits with the specified exit code",
	params = "<code>"
})

lush.Command:new(
{
	name = "vecdesc",
	validate_arguments =
	{
		{
			name = "vector",
			type = "vector",
			single = true
		}
	},
	callback = function(context, args, options)
		context.stdout:push("x: " .. tostring(args["vector"].x))
		context.stdout:push("y: " .. tostring(args["vector"].y))
		context.stdout:push("z: " .. tostring(args["vector"].z))
	end,
	privs = {},
	description = "verbosely prints information about a vector",
	params = "<vec>"
})

lush.Command:new(
{
	name = "lush",
	validate_arguments =
	{
		{
			name = "code",
			type = "string",
			single = true,
			optional = true
		}
	},
	callback = function(context, args, options)
		lush.Parser.run_shell(args["code"], lush.ShellContext:new(lush.global_env))
	end,
	privs = {},
	description = "runs a shell",
	params = "<code>"
})

lush.Command:new(
{
	name = "export",
	allowed_options =
	{
		["read-only"] = false,
		["volatile"] = false,
	},
	validate_arguments =
	{
		{
			name = "param",
			type = "string",
			single = true
		}
		-- manually checking the last argument
		-- @todo implement this as an optional argument when rules support it
	},
	callback = function(context, args, options)
		if not args[1] then
			lush.global_env:unset(args["param"])
		else
			lush.global_env:set(args["param"], args["value"], options["read-only"], options["volatile"])
		end
	end,
	privs =
	{
		export = true
	},
	description = "exports the parameter to the global environment, where every player can access it. if no value is provided, it unsets the global variable instead",
	params = "<parameter name> <value>"
})

lush.Command:new(
{
	name = "env",
	callback = function(context, args, options)
		if #args ~= 0 then
			error("invalid arguments: too many arguments")
		end

		local function print_env(context, env)
			if env.inherited_env.instance_of == "Env" then
				print_env(context, env.inherited_env)
			end

			for parameter, value in pairs(env.parameters) do
				if type(value) == "function" then
					context.stdout:push(parameter .. "=" .. lush.DatatypeValidator.cast_to(value(env), "string"))
				else
					context.stdout:push(parameter .. "=" .. lush.DatatypeValidator.cast_to(value, "string"))
				end
			end

			for parameter, value in pairs(env.volatile_parameters) do
				if type(value) == "function" then
					context.stdout:push(parameter .. "=" .. lush.DatatypeValidator.cast_to(value(env), "string"))
				else
					context.stdout:push(parameter .. "=" .. lush.DatatypeValidator.cast_to(value, "string"))
				end
			end
		end

		print_env(context, context.env)
	end,
	privs = {},
	description = "prints the entire environment to STDOUT",
	params = ""
})
