-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

minetest.unregister_chatcommand("mods")

minetest.register_privilege("lush_heal",
{
	description = "whether the player can use the heal command",
	give_to_singleplayer = false
})

minetest.register_privilege("lush_damage",
{
	description = "whether the player can use the damage/kill command",
	give_to_singleplayer = false
})

minetest.register_privilege("invisible",
{
	description = "whether the player can use the invis command",
	give_to_singleplayer = false
})

lush.Command:new(
{
	name = "days",
	allowed_options = {},
	validate_arguments = {},
	callback = function(context, args, options)
		context.stdout:push(minetest.get_day_count())
	end,
	privs = {},
	description = "prints the amounts of days passed",
	params = ""
})

lush.Command:new(
{
	name = "set_node",
	validate_arguments = 
	{
		{
			name = "pos",
			type = "vector",
			single = false
		},
		{
			name = "node",
			type = "node_name",
			single = true
		},
	},
	callback = function(context, args, options)
		minetest.debug(dump(args))
		for i, pos in args["pos"]:iterator() do
			minetest.set_node(pos, {name = args["node"]})
		end
	end,
	privs = {give = true},
	description = "sets a node",
	params = "<pos> <node name>"
})

local storage = minetest.get_mod_storage()

local time_update_job

local function time_update_func(time)
	minetest.set_timeofday(time)
	time_update_job = minetest.after(1, time_update_func, time)
end

if storage:contains("lush_permament_time") and storage:get_float("lush_permament_time") ~= -1 then
	time_update_job = minetest.after(1, time_update_func, storage:get_float("lush_permament_time"))
end

lush.Command:new(
{
	name = "time",
	allowed_options = {["permament"] = false},
	validate_arguments =
	{
		{
			name = "time",
			type = {"time", "string"},
			single = true
		}
	},
	callback = function(context, args, options)
		local time_of_day = 0

		if type(args["time"]) == "string" then
			if args["time"] == "midnight" then
				time_of_day = 0
			elseif args["time"] == "night" then
				time_of_day = 0.8
			elseif args["time"] == "sunset" then
				time_of_day = 0.75
			elseif args["time"] == "noon" then
				time_of_day = 0.5
			elseif args["time"] == "day" then
				time_of_day = 0.3
			elseif args["time"] == "sunrise" then
				time_of_day = 0.2
			else
				error(string.format("%s isnt a recgonized time", args["time"]))
			end
		else
			if args["time"] >= 86400 or args["time"] < 0 then
				error("time must be between 0 seconds and 24 hours")
			end

			time_of_day = args["time"] / 86400
		end

		minetest.set_timeofday(time_of_day)

		if options["permament"] then
			time_update_job = minetest.after(1, time_update_func, time_of_day)
			storage:set_float("lush_permament_time", time_of_day)
		else
			time_update_job:cancel()
			storage:set_float("lush_permament_time", -1)
		end
	end,
	privs = {settime = true},
	description = "sets time",
	params = "<time>"
})

lush.Command:new(
{
	name = "lunch",
	allowed_options = {},
	validate_arguments =
	{
		{
			name = "objects",
			type = "objectref",
			single = false
		},
		{
			name = "vector",
			type = "vector",
			single = true
		}
	},
	callback = function(context, args, options)
		for i, v in args["objects"]:iterator() do
			v:add_velocity(args["vector"])
		end
	end,
	privs =
	{
		lush_damage = true
	},
	description = "adds velocity to entities",
	params = "<selector> <vector>"
})

lush.Command:new(
{
	name = "teleport",
	allowed_options = {},
	validate_arguments =
	{
		{
			name = "objects",
			type = "objectref",
		},
		{
			rule_type = "terminal",
			value = "to",
			optional = true,
		},
		{
			name = "pos",
			type = "vector",
			single = true
		}
	},
	callback = function(context, args, options)
		local lm = 31007 -- equals MAX_MAP_GENERATION_LIMIT in C++
		local pos = args["pos"]
		if pos.x < -lm or pos.x > lm or pos.y < -lm or pos.y > lm
				or pos.z < -lm or pos.z > lm then
			error("position out of map's bounds")
		end

		if (context.env:get("player") ~= args["objects"]:get(1):get_player_name() or #args["objects"] > 1) and not minetest.check_player_privs(context.env:get("player"), "bring") then
			error("you need the 'bring' privilege to teleport anything other than yourself")
		end

		for i, v in args["objects"]:iterator() do
			if v:get_pos() and not v:get_attach() then
				v:set_pos(args["pos"])
			end
		end

	end,
	privs = {teleport = true},
	description = "teleports entities to position",
	params = "<selector> [to] <pos>"
})

lush.Command:new(
{
	name = "kill",
	allowed_options = {},
	validate_arguments =
	{
		{
			name = "objects",
			type = "objectref",
		},
	},
	callback = function(context, args, options)
		for i, v in args["objects"]:iterator() do
			lush.helpers.deal_damage(v, v:get_hp(), nil, true)
		end

	end,
	privs = {},
	description = "kills entitiy/s",
	params = "<selector>"
})

lush.Command:new(
{
	name = "damage",
	allowed_options =
	{
		["type"] = true,
	},
	validate_arguments =
	{
		{
			name = "targets",
			type = "objectref",
			single = false,
		},
		{
			name = "amount",
			type = "number",
			single = true,
		},
	},
	callback = function(context, args, options)
		local damage_type = "set_hp" or options["type"]

		if type(damage_type) ~= "string" then
			error("damage type should be a string")
		end

		for i, v in args["targets"]:iterator() do
			lush.helpers.deal_damage(v, args["amount"], nil, true)
		end

	end,
	privs = {},
	description = "deals damage to entities",
	params = "<target> <amount>"
})

lush.Command:new(
{
	name = "spawnpoint",
	allowed_options = {},
	validate_arguments =
	{
		{
			name = "names",
			type = "player_name",
			single = false,
		},
		{
			name = "pos",
			type = "vector",
			single = true,
		},
	},
	callback = function(context, args, options)
		for _, v in args["names"]:iterator() do
			lush.spawnpoints[v] = args["pos"]
		end

		lush.Storage.save("spawnpoints", lush.spawnpoints)
	end,
	privs = {},
	description = "sets the respawn point for the player",
	params = "<target> <pos>"
})

lush.Command:new(
{
	name = "clear_spawnpoint",
	allowed_options = {},
	validate_arguments =
	{
		{
			name = "names",
			type = "player_names",
			single = false,
		},
	},
	callback = function(context, args, options)
		for _, v in args["names"]:iterator() do
			lush.spawnpoints[v] = nil
		end

		lush.Storage.save("spawnpoints", lush.spawnpoints)
	end,
	privs = {},
	description = "removes the previously set spawnpoint",
	params = "<target> <pos>"
})

lush.Command:new(
{
	name = "heal",
	allowed_options = {},
	validate_arguments =
	{
		{
			name = "targets",
			type = "objectref",
			single = false,
		},
		{
			name = "amount",
			type = "number",
			single = true,
		},
	},
	callback = function(context, args, options)
		for i, v in args["targets"]:iterator() do
			lush.helpers.deal_damage(v, -args["amount"], nil, false)
		end
	end,
	privs =
	{
		lush_heal = true
	},
	description = "heals entities by an amount",
	params = "<target> <amount>"
})

lush.Command:new(
{
	name = "list",
	allowed_options = {},
	validate_arguments = {},
	callback = function(context, args, options)
		for _, player in ipairs(minetest.get_connected_players()) do
			context.stdout:push(player:get_player_name())
		end
	end,
	privs =
	{
		debug = true
	},
	description = "lists online players",
	params = ""
})

lush.Command:new(
{
	name = "seed",
	allowed_options = {},
	validate_arguments = {},
	callback = function(context, args, options)
		context.stdout:push(minetest.get_mapgen_setting("seed"))
	end,
	privs =
	{
		debug = true
	},
	description = "prints the world's seed",
	params = ""
})

lush.Command:new(
{
	name = "lsmods",
	allowed_options = {},
	validate_arguments = {},
	callback = function(context, args, options)
		for i, v in ipairs(minetest.get_modnames()) do
			context.stdout:push(v)
		end
	end,
	privs =
	{
		debug = true
	},
	description = "prints the world's seed",
	params = ""
})

-- credit to SkyBuilder1717's Essentials mod
-- from whom this implementation is partly ripped
lush.Command:new(
{
	name = "invis",
	validate_arguments =
	{
		{
			type = "player",
			name = "players",
			single = false
		},
		{
			type = "string",
			name = "is_enabled",
			single = true
		}
	},
	callback = function(context, args, options)
		local prop

		if args["is_enabled"] == "on" then
			prop = {
				visual_size = {x = 0, y = 0, z = 0},
				is_visible = false,
				nametag_color = {r=255,g=255,b=255,a=255},
				pointable = false,
				makes_footstep_sound = false,
				show_on_minimap = false,
			}
		elseif args["is_enabled"] == "off" then
			prop = {
				visual_size = {x = 1, y = 1, z = 1},
				is_visible = true,
				nametag_color = {r=255,g=255,b=255,a=0},
				pointable = true,
				makes_footstep_sound = true,
				show_on_minimap = true,
			}
		else
			error("invalid argument: '" .. tostring(args["is_enabled"]) .. "', should be either on or off")
		end

		for i, v in args["players"]:iterator() do
			v:set_properties(prop)

			if args["is_enabled"] == "on" then
				v:get_meta():set_int("invisible", 1)
			elseif args["is_enabled"] == "off" then
				v:get_meta():set_int("invisible", 0)
			end
		end
	end,
	privs =
	{
		invisible = true
	},
	description = "makes the entities invisible",
	params = "<players>"
})

lush.Command:new(
{
	name = "give",
	allowed_options =
	{
		["wear"] = true,
		["amount"] = true
	},
	validate_arguments =
	{
		{
			type = "player",
			name = "players",
			single = false
		},
		{
			type = "string",
			name = "item_name",
			single = true
		},
		{
			type = "table",
			name = "meta",
			single = true
		}
	},
	callback = function(context, args, options)
		local stack = ItemStack({name = args["item_name"]})

		local meta = stack:get_meta()
		meta:from_table({fields = args["meta"]})

		if options["wear"] then
			stack:set_wear(options["wear"])
		end

		if options["amount"] then
			stack:set_count(options["amount"])
		end

		local inv
		for i, object in args["players"]:iterator() do
			inv = object:get_inventory()

			inv:add_item("main", stack)
		end

	end,
	privs =
	{
		invisible = true
	},
	description = "makes the entities invisible",
	params = "<target>"
})

lush.Command:new(
{
	name = "nodeinfo",
	allowed_options =
	{
		["human"] = false
	},
	validate_arguments =
	{
		{
			rule_type = "terminal",
			name = "type",
			value =
			{
				["metadata"] = true,
				["inventory"] = true,
				["groups"] = true,
				["name"] = true,
				["param1"] = true,
				["param2"] = true,
			}
		},
		{
			name = "pos",
			type = "vector",
			single = true
		}
	},
	callback = function(context, args, options)

		if args["type"] == "metadata" then
			local meta = minetest.get_meta(args["pos"]):to_table()

			for k, v in pairs(meta.fields) do
				context.stdout:push(tostring(k) .. "=" .. tostring(v))
			end
		elseif args["type"] == "inventory" then
			local inv = minetest.get_inventory({type = "node", pos = args["pos"]})

			if inv then
				if options["human"] then
					-- print in tree like format
					-- the code looks funny because the last element must start with
					-- an "└─ " even if the last item in the inventory is empty
					-- this is a very clever solution to this problem
					for list_name, list in pairs(inv:get_lists()) do
						context.stdout:push(list_name)

						local previous_item
						for i, item in pairs(list) do

							if not item:is_empty() then
								if previous_item then
									context.stdout:push("├─ " .. previous_item:to_string())
								end
								previous_item = item
							end
						end

						if previous_item then
							context.stdout:push("└─ " .. previous_item:to_string())
						end
					end
				else
					for list_name, list in pairs(inv:get_lists()) do
						context.stdout:push("List " .. list_name)
						for _, v in pairs(list) do
							context.stdout:push(v:to_string())
						end
						context.stdout:push("EndInventoryList")
					end
				end
			end
		elseif args["type"] == "groups" then
			local nodedef = minetest.registered_nodes[minetest.get_node(args["pos"]).name]

			for k, v in pairs(nodedef.groups) do
				context.stdout:push(tostring(k) .. "=" .. tostring(v))
			end
		elseif args["type"] == "name" then
			context.stdout:push(minetest.get_node(args["pos"]).name)
		elseif args["type"] == "param1" then
			context.stdout:push(minetest.get_node(args["pos"]).param1)
		elseif args["type"] == "param2" then
			context.stdout:push(minetest.get_node(args["pos"]).param2)
		end
	end,
	privs =
	{
		debug = true
	},
	description = "prints information about the node at a given position",
	params = "<pos>"
})
