-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod Command
-- this class defines how commands are executed
local helpers = require("helpers")
local DatatypeValidator = require("datatype_validator")
local TypedList = require("typed_list")

local inspect = require("inspect")

local Registered_Commands = {}

local Command =
{
	instance_of = "Command",
}

Command.__index = Command

--- constructor <br>
--	note: dosent create a commands if there already exists a commands with the same name
--	@tab t definition table with matching keys as the class
--	@treturn Command|nil the newly created Command instance
--	@see helpers.is_option
function Command:new(t)
	if t == nil then
		t = {}
	end

	-- {{{ type checking
		if type(t) ~= "table" then error("t should be of type table. instead of " .. type(t)) end
		if t.name and type(t.name) ~= "string" then error("name has to be of type string") end
		if t.allowed_options and type(t.allowed_options) ~= "table" then error("name has to be of type table") end
		if t.callback and type(t.callback) ~= "function" then error("name has to be of type function") end
	-- }}}

	self = setmetatable(
	{
		--- @string[opt=""] name name of the command
		name = t.name or "",

		--- @func[opt=nil]
		--	callback callback function that is called when the command is executed <br>
		--	should be called with call() method
		--  @ShellContext context shell context for the execution
		--  @tab args list of string arguments
		--  @tab options dictionary where the key is a string name of the option and the value is the argument
		--  (if applicable, otherwise is false)
		--  @treturn ShellContext shell context after execution
		--  @see Command:call
		callback = t.callback or nil,

		--- @tab[opt={}] allowed_options dictionary represending what options are allowed <br>
		-- key - string character <br>
		-- value - boolean value, whether it takes an argument <br>
		allowed_options = t.allowed_options or {},

		--- @tab[opt={}] validate_arguments a list that describes what arguments the program expects <br>
		--	each element is a table with the following members: <br>
		--	name: the name of the argument, so it can be accessed by name instead of an index
		--	type: string or table of the datatypes it expects <br>
		--	single: whether is a single value or a TypedList
		--	terminator: string or a set of terminators that the argument should equal <br>
		--	consume_terminator: if true, then the terminator is removed from the argument list (defaults to true) <br>
		--
		--	if validate_arguments isnt defined, then no argument checking is done
		 validate_arguments = t.validate_arguments or nil,

		 --- @string[opt=nil] the datatype the stdin is expected to be, or nil if no checking
		 stdin_type = t.stdin_type or nil
	}, Command)

	if not Command.is_name_registered(self.name) then
		-- print(inspect(Registered_Commands))
		-- print(inspect(self.name))
		Registered_Commands[self.name] = self
	else
		-- print(inspect(Registered_Commands))
		error(string.format("trying to create a new command with the same name as a previously defined command: %s", self.name))
	end
	return self
end

--- splits arguments into options and executes the command
-- @ShellContext context shell context for the execution
-- @tab args list of arguments and options
-- @treturn ShellContext context after the execution
function Command:call(context, args)
	-- {{{ type checking
	if context.instance_of ~= "ShellContext" then error("context should be an instance of ShellContext. instead of " .. context.instance_of) end
	-- }}}

	local options = {}
	local validated_args = {}
	local succes
	local skip = false

	if self.stdin_type and context.stdin.type ~= self.stdin_type then
		local success	= context.stdin:cast_to(self.stdin_type)

		if not success then
			error("stdin is incorrect type: expected \"" .. self.stdin_type .. "\" but got \"" .. DatatypeValidator.get_type_of(self.stdin) .. "\"")
		end
	end

	-- maybe all this logic should be moved to CommandInvocation?
	local is_valid_option, option_name
	for i, arg in ipairs(args) do
		if not skip then
			is_valid_option = false
			if type(arg) == "string" then
				is_valid_option, option_name = helpers.is_option(arg)
				if is_valid_option then

					if self.allowed_options[option_name] == nil then
						error(string.format("unrecognized option '%s'", option_name))

						return context
					end

					if(self.allowed_options[option_name]) == true then
						-- if takes an argument

						if i == #args then
							-- if the option is the last argument, therefore there is no argument for the option passed
							error(string.format("option '%s' expects an argument", option_name))
						end

						options[option_name] = args[i + 1]
						skip = true
					else
						options[option_name] = true
					end
				end
			end

			if not is_valid_option then
				table.insert(validated_args, arg)
			end
		else
			skip = false
		end
	end

	if self.validate_arguments then
		validated_args = DatatypeValidator.validate(validated_args, self.validate_arguments)
	end

	local success, message = pcall(self.callback, context, validated_args, options)

	if success == true then
		context.exit_status = 0
	else
		message = message or "no error message provided"
		error(message)
	end

	return context
end

--- unregisters the command and destroys the instance
--  @return nil
--  @see Command.is_name_registered
function Command:unregister()
	Registered_Commands[self.name] = nil
	self = nil
end

--- returns whether the command with the specified name is registered
--  @string name name of the command
--  @treturn bool whether the command is already registered
--  @see Command:unregister, Command:get_command_by_name
function Command.is_name_registered(name)
	return Registered_Commands[name] ~= nil
end

--- returns a registered command with the matching name
--  @string name name of the command
--  @treturn Command the command
--  @see Command:is_name_registered
function Command.get_command_by_name(name)
	return Registered_Commands[name]
end

--- creates an alias for a command
--	@string name
--	@string new_alias
--	@treturn nil
function Command.alias(name, new_alias)
	Registered_Commands[new_alias] = Registered_Commands[name]
end

return Command, Registered_Commands
