-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

minetest.unregister_chatcommand("grantme")
minetest.unregister_chatcommand("revokeme")

lush.Command:new(
{
	name = "ban",
	allowed_options =
	{
		["script"] = false,
	},
	validate_arguments =
	{
		{
			name = "players",
			type = "player_name",
			single = false
		}
	},
	callback = function(context, args, options)
		if minetest.is_singleplayer() then
			error("cant ban players in singleplayer")
		end

		local success
		local successful_bans = {}
		local failed_bans = {}
		for _, name in pairs(args["players"]) do
			success = minetest.ban_player(name)

			if success then
				table.insert(successful_bans, name)
			else
				table.insert(failed_bans, name)
			end
		end

		if #successful_bans == 0 then
			error("failed to ban players")
		end

		if options["script"] then
			context.stdout:concat(successful_bans)
		else
			if #failed_bans ~= 0 then
				context.stdout:push(string.format("failed to ban players: %s", table.concat(failed_bans, ", ")))
			end
		end

		minetest.log("action", string.format("%s banned players: %s", context.env:get("player"), table.concat(successful_bans, ", ")))
	end,
	privs =
	{
		ban = true
	},
	description = "bans players",
	params = "<players>"
})

lush.Command:new(
{
	name = "privs",
	allowed_options = {},
	validate_arguments =
	{
		{
			name = "player",
			type = "player_name",
			single = true
		}
	},
	callback = function(context, args, options)
		for priv, _ in pairs(minetest.get_player_privs(args["player"])) do
			context.stdout:push(priv)
		end
	end,
	privs =
	{
		debug = true
	},
	description = "lists player's privileges",
	params = "<player>"
})

lush.Command:new(
{
	name = "lsbans",
	allowed_options = {},
	validate_arguments = {},
	callback = function(context, args, options)
		context.stdout:concat(minetest.get_ban_list())
	end,
	privs = {ban = true},
	description = "lists all banned players",
	params = ""
})

lush.Command:new(
{
	name = "unban",
	allowed_options =
	{
		["script"] = false,
	},
	validate_arguments =
	{
		{
			name = "players",
			type = "player_name",
			single = false
		}
	},
	callback = function(context, args, options)
		local success
		local successful_unbans = {}
		local failed_unbans = {}
		for _, name in pairs(args["players"]) do
			success = minetest.unban_player_or_ip(name)

			if success then
				table.insert(successful_unbans, name)
			else
				table.insert(failed_unbans, name)
			end
		end

		if #successful_unbans == 0 then
			error("failed to unban players")
		end

		if options["script"] then
			context.stdout:concat(successful_unbans)
		else
			if #failed_unbans ~= 0 then
				context.stdout:push(string.format("failed to unban players: %s", table.concat(failed_unbans, ", ")))
			end
		end

		minetest.log("action", string.format("%s unbanned players: %s", context.env:get("player"), table.concat(successful_unbans, ", ")))
	end,
	privs = {ban = true},
	description = "unbans players",
	params = "<player name> | <IP adress>"
})

lush.Command:new(
{
	name = "kick",
	{
		["script"] = false
	},
	validate_arguments =
	{
		{
			name = "players",
			type = "player_name",
			single = false,
		},
		{
			name = "reason",
			type = "string",
			single = true
		},
	},
	callback = function(context, args, options)
		if minetest.is_singleplayer() then
			error("cant kick players in singleplayer")
		end

		local success
		local successful_kicks = {}
		local failed_kicks = {}
		for _, name in pairs(args["players"]) do
			success = minetest.kick_player(name, args["reason"])

			if success then
				table.insert(successful_kicks, name)
			else
				table.insert(failed_kicks, name)
			end
		end

		if #successful_kicks == 0 then
			error("failed to kick players")
		end

		if options["script"] then
			context.stdout:concat(successful_kicks)
		else
			if #failed_kicks ~= 0 then
				context.stdout:push(string.format("failed to ban players: %s", table.concat(failed_kicks, ", ")))
			end
		end

		minetest.log("action", string.format("%s banned players: %s", context.env:get("player"), table.concat(successful_kicks, ", ")))
	end,
	privs = {ban = true},
	description = "kicks player out of the server",
	params = "<players>"
})

lush.Command:new(
{
	name = "grant",
	allowed_options = {},
	validate_arguments =
	{
		{
			name = "player",
			type = "player_name",
			single = true
		},
		{
			rule_type = "terminal",
			value =
			{
				["privs"] = true,
				["privileges"] = true,
			},
			consume = true,
			optional = true,
		},
		{
			rule_type = "block",
			name = "privs",
			type = "string",
			required = true
		},
	},
	callback = function(context, args, options)
		local caller_privs = minetest.get_player_privs(context.env:get("player"))

		if not (caller_privs.privs or caller_privs.basic_privs) then
			error("insufficient privileges")
		end

		local player_privs = minetest.get_player_privs(args["player"])
		local unknown_privs = {}
		local basic_privs = minetest.string_to_privs(minetest.settings:get("basic_privs") or "interact,shout")

		for _, priv in pairs(args["privs"]) do
			if not basic_privs[priv] and not caller_privs.privs then
				error("Your privileges are insufficient")
			end
			if not minetest.registered_privileges[priv] then
				table.insert(unknown_privs, priv)
			end
			player_privs[priv] = true
		end

		if #unknown_privs ~= 0 then
			error("unknown privileges: " .. table.concat(unknown_privs, ", "))
		end

		minetest.set_player_privs(args["player"], player_privs)

		for _, priv in pairs(args["privs"]) do
			-- call the on_grant callbacks
			minetest.run_priv_callbacks(args["player"], priv, context.env:get("player"), "grant")
		end

		minetest.log("action", context.env:get("player") .. ' granted (' .. table.concat(args["privs"], ", ") .. ') privileges to ' .. args["player"])

		if args["player"] ~= context.env:get("player") then
			minetest.chat_send_player(args["player"],
				context.env:get("player") .. " granted privileges" .. table.concat(args["privs"], ", ") .. " to you")
		end
	end,
	privs = {},
	description = "grants privileges to player",
	params = "<player> [privs | privileges] <priv> ..."
})

lush.Command:new(
{
	name = "revoke",
	allowed_options = {},
	validate_arguments =
	{
		{
			name = "player",
			type = "player_name",
			single = true
		},
		{
			rule_type = "terminal",
			value =
			{
				["privs"] = true,
				["privileges"] = true,
			},
			consume = true,
			optional = true,
		},
		{
			rule_type = "block",
			name = "privs",
			type = "string",
			required = true
		},
	},
	callback = function(context, args, options)
		local caller_privs = minetest.get_player_privs(context.env:get("player"))

		if not (caller_privs.privs or caller_privs.basic_privs) then
			error("insufficient privileges")
		end

		local player_privs = minetest.get_player_privs(args["player"])
		local unknown_privs = {}
		local basic_privs = minetest.string_to_privs(minetest.settings:get("basic_privs") or "interact,shout")
		local is_admin = not minetest.is_singleplayer()
			and args["player"] == minetest.settings:get("name")
			and args["player"] ~= ""
		local priv_def

		for _, priv in pairs(args["privs"]) do
			if not basic_privs[priv] and not caller_privs.privs then
				error("Your privileges are insufficient")
			end

			priv_def = minetest.registered_privileges[priv]

			if not priv_def then
				if not player_privs[priv] then
					table.insert(unknown_privs, priv)
				end
			elseif (minetest.is_singleplayer() and priv_def.give_to_singleplayer)
				or is_admin and priv_def.give_to_admin then
				error("you cant revoke " .. priv .. " from admin")
			end
			player_privs[priv] = nil
		end

		if #unknown_privs ~= 0 then
			error("unknown privileges: " .. table.concat(unknown_privs, ", "))
		end

		minetest.set_player_privs(args["player"], player_privs)

		for _, priv in pairs(args["privs"]) do
			-- call the on_grant callbacks
			minetest.run_priv_callbacks(args["player"], priv, context.env:get("player"), "revoke")
		end

		minetest.log("action", context.env:get("player") .. ' revoked (' .. table.concat(args["privs"], ", ") .. ') privileges from ' .. args["player"])

		if args["player"] ~= context.env:get("player") then
			minetest.chat_send_player(args["player"],
				context.env:get("player") .. " granted privileges" .. table.concat(args["privs"], ", ") .. " to you")
		end
	end,
	privs = {},
	description = "takes away privileges from a player",
	params = "<player> [privs | privileges] <priv> ..."
})

