-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @module DatatypeValidator
-- helper functions related to casting and reading datatypes of values
-- aswell as rule based argument validation

local TypedList

local DatatypeValidator = {}

local registered_types = {}

function DatatypeValidator.register_type(name, def)
	registered_types[name] = def
end

function DatatypeValidator.unregister_type(name)
	registered_types[name] = nil
end

--- returns the datatype of the given argument
--	@param value
--	@treturn string|nil returns nil when cant decide what datatype it is
function DatatypeValidator.get_type_of(value)
	if type(value) ~= "table" and type(value) ~= "userdata" then
		return type(value)
	end

	for type_name, type_def in pairs(registered_types) do
		if type_def.get_type(value) then
			return type_name
		end
	end

	if type(value) == "table" then
		return "table"
	end
end

--- returns whether the datatype is a valid datatype for lush
--	@param value
--	@treturn bool
function DatatypeValidator.is_valid_type(value)
	return DatatypeValidator.get_type_of(value) ~= nil
end

--- tries to convert the given value to the datype
--	@param value the value to convert
--	@string type the datatype to convert to
--	@treturn bool whether sucessful
--	@return the converted value, or nil if cant convert
function DatatypeValidator.cast_to(value, type)
	local current_type = DatatypeValidator.get_type_of(value)

	if type == "table" then
		return nil, false
	end

	if current_type == type then
		return value, true
	end

	if registered_types[current_type] and registered_types[current_type].conversions[type] then
		local converted_value = registered_types[current_type].conversions[type](value)
		return converted_value, converted_value ~= nil
	end

	return nil, false
end

--- same as cast_to(), except it converts every element in the table
--	@tab t table whose elements should be converted
--	@string type the datatype to convert to
--	@treturn tab a table with converted elements
--	@treturn bool whether sucessful
--	@see DatatypeValidator.cast_to
function DatatypeValidator.cast_list_to(t, type)
	local current_type = DatatypeValidator.get_type_of(t[1])

	if type == "table" then
		return nil, false
	end

	local converted_table = {}
	local converted_value

	if #t == 0 then
		return {}, true
	end

	if current_type == type then
		-- copying the table
		for i, v in pairs(t) do
			converted_table[i] = v
		end

		return converted_table, true
	end

	for i, v in pairs(t) do
		if registered_types[current_type].conversions[type] then
			converted_value = registered_types[current_type].conversions[type](v)
		end

		if converted_value then
			table.insert(converted_table, converted_value)
		end
	end

	return converted_table, #converted_table ~= 0
end

-- a rule is a table with specific members that describe:
-- * check if the input table is correct
-- * the type of elements in the table
-- * whether is optional or not
-- * how the output table is structured
--
-- each rule has a type and the type specifies what members are accepted and how the rule is interpreted
-- defaults to "value"
-- the supported types and their members are:
--
-- value:
-- this rule type describes a value in the input table that has a certain type
-- and is put in the output table by a human readable string key
-- * name - string - the index of this value in the output table
-- * type - string - the type that the value must have
-- * optional - bool - whether the value is optional [default: false] not implemented yet!!!
--   only works if any of the conditions are met:
--   * the next rule has a different datatype
--   * every rule afterwards is optional
-- * single - bool - whether is expected to be a table of values or a single value [default: true]
--
-- terminal:
-- this rule type describes a string value that are required/optional to be there in order to match
-- * value - string or set of strings - what the terminator must equal
-- * name - string - the index of the terminator in the output table, only considered if consume flag is false
-- * consume - bool - if true then the terminator isnt put in the output table [default: true]
-- * optional - bool - whether the terminator is optional [default: false]
--
-- block:
-- this rules type describes a list of values that are matched up until value dosent match the criteria
-- * name - string - the key by which the list of values can be accessed in the output table
-- * type - string - the type each value must match
-- * terminator - another rule whose type is "terminal" - any element that matches this terminal ends the block
-- * required - bool - if true, at least one value must be matched [default: true]

--- validates elements of the given table according to the rules
--	@tab t table to validate
--	@tab rules list of rules to apply
--	@treturns table a new table with rules applied
function DatatypeValidator.validate(t, rules)
	-- {{{ type checking
	if type(t) ~= "table" then error("t should be of type table. instead of " .. type(t)) end
	if type(rules) ~= "table" then error("rules should be of type table. instead of " .. type(rules)) end
	-- }}}

	TypedList = TypedList or require("typed_list")
	-- minetest.debug("validating args", dump(t))
	-- minetest.debug("rules", dump(rules))
	-- minetest.debug("applying rules")

	local output = {}
	local i = 1
	local value
	local is_valid_terminator
	local is_valid_type 
	local current_type
	local successfully_converted 

	for _, rule in pairs(rules) do
		-- minetest.debug("rule", dump(rule))
		if rule.rule_type == "terminal" then
			if type(t[i]) ~= "string" then
				is_valid_terminator = false
			elseif type(rule.value) == "string" then
				is_valid_terminator = t[i] == rule.value
			else
				is_valid_terminator = rule.value[t[i]]
			end

			if not is_valid_terminator then
				if not rule.optional then
					error("#" .. tostring(i) .. " argument is invalid" )
				end
			elseif not rule.consume then
				if rule.name then
					output[rule.name] = t[i]
				else
					table.insert(output, t[i])
				end
				i = i + 1

			end
		elseif rule.rule_type == "value" or rule.rule_type == nil then
			-- minetest.debug("validating a value: #" .. tostring(i))

			if rule.single then
				if type(t[i]) == "table" and t[i].instance_of == "TypedList" then
					-- minetest.debug("expected a single value, got a list")
					value = t[i]:get(1)
				else
					-- minetest.debug("expected a single value, got a single value")
					value = t[i]
				end

				current_type = DatatypeValidator.get_type_of(value)
			else
				-- minetest.debug(dump(t[i]))
				if type(t[i]) ~= "table" or (type(t[i]) == "table" and t[i].instance_of ~= "TypedList") then
					-- minetest.debug("expected a list, got a single value")
					value = TypedList:new()
					value:push(t[i])
				else
					-- minetest.debug("expected a list, got list")
					value = t[i]
				end

				current_type = value.type
			end

			-- minetest.debug("current value", dump(value))
			-- minetest.debug("current type", current_type)

			if type(rule.type) == "string" then
				is_valid_type = current_type == rule.type
			else
				is_valid_type = rule.type[current_type]
			end

			if not is_valid_type then
				-- trying to convert it to a correct type
				if type(rule.type) == "string" then
					-- when there is a single type to convert to
					if type(value) == "table" and value.instance_of == "TypedList" then
						successfully_converted = value:cast_to(rule.type)
					else
						value, successfully_converted = DatatypeValidator.cast_to(value, rule.type)
					end
				else
					-- when there are multiple accepted types
					-- looping over all accepted datatypes and trying to convert untill any conversion is successful
					-- @todo i have a gut feeling this is broken
					local type_index = 1
					local converted_value
					while not successfully_converted and type_index <= #rule.type do
						if type(value) == "table" and value.instance_of == "TypedList" then
							-- when rule.type is a TypedList
							successfully_converted = value:cast_to(rule.type[type_index])
						else
							-- when rule.type is a single string
							converted_value, successfully_converted = DatatypeValidator.cast_to(value, rule.type[type_index])
						end
						type_index = type_index + 1
					end

					if successfully_converted then
						value = converted_value
					end
				end

				if not successfully_converted then
					error("#" .. tostring(i) .. " argument is invalid datatype, is " .. tostring(current_type) .. ", but should be " .. dump(rule.type))
				end
			end

			if rule.name then
				output[rule.name] = value
			else
				table.insert(output, value)
			end

			i = i + 1

		elseif rule.rule_type == "block" then
			local block_values = {}

			while true do
				-- checking if is a terminator
				if rule.terminator then
					if type(rule.terminator.value) == "string" then
						is_valid_terminator = t[i] == rule.terminator.value
					else
						is_valid_terminator = rule.terminator.value[t[i]]
					end

					if is_valid_terminator then
						if not rule.terminator.consume then
							if rule.terminator.name then
								output[rule.terminator.name] = t[i]
							else
								table.insert(output, t[i])
							end
						end

						i = i + 1
						break
					end
				end

				-- checking whether has a valid type
				if type(t[i]) == "table" and t[i].instance_of == "TypedList" then
					current_type = t[i].type
				else
					current_type = DatatypeValidator.get_type_of(t[i])
				end

				if type(rule.type) == "string" then
					is_valid_type = current_type == rule.type
				else
					is_valid_type = rule.type[current_type]
				end

				if not is_valid_type then
					break
				end

				table.insert(block_values, t[i])
				i = i + 1
			end

			if rule.name then
				output[rule.name] = block_values
			else
				table.insert(output, block_values)
			end

		end
	end

	-- if there are more arguments than the ones checked
	if #t >= i then
		error("invalid arguments: too many arguments ")
	end

	-- minetest.debug("end of applying rules")

	return output
end

return DatatypeValidator
