-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local DatatypeValidator = require("datatype_validator")
local helpers = require("helpers")

DatatypeValidator.register_type("string", 
{
	get_type = function(value)
		return type(value) == "string"
	end,

	conversions = 
	{
		["node_name"] = function(value)
			return helpers.normalize_nodename(value)
		end,
		
		["number"] = function(value)
			return tonumber(value)
		end,

		["time"] = function(value)
			return helpers.parse_time(value)
		end,

		["vector"] = function(value)
 			converted_value = vector.from_string(value)
 			if converted_value then
 				return converted_value, true
 			elseif minetest.get_player_by_name(value) then
 				return minetest.get_player_by_name(value):get_pos(), true
 			end
		end,

		["objectref"] = function(value)
			return minetest.get_player_by_name(value)
		end,

		["player"] = function(value)
			return minetest.get_player_by_name(value)
		end,

		["player_name"] = function(value)
			if minetest.get_player_by_name(value) then
				return value
			end
		end
	},
})

DatatypeValidator.register_type("node_name", 
{
	get_type = function(value)
		return type(value) == "string" and helpers.normalize_nodename(value) ~= nil
	end,

	conversions = 
	{
		["string"] = function(value)
			return value
		end,
	},
})

DatatypeValidator.register_type("number", 
{
	get_type = function(value)
		return type(value) == "number"
	end,

	conversions = 
	{
		["string"] = function(value)
			return tostring(value)
		end,

		["time"] = function(value)
			return value
		end,
	},
})

DatatypeValidator.register_type("time", 
{
	get_type = function(value)
		return type(value) == "number"
	end,

	conversions = 
	{
		["string"] = function(value)
			return helpers.time_to_string(value)
		end,

		["number"] = function(value)
			return value
		end,
	},
})

DatatypeValidator.register_type("range",
{
	get_type = function(value)
		return type(value) == "table" and value.instance_of == "Range"
	end,

	conversions = 
	{
		["string"] = function(value)
			return tostring(value)
		end,
	},
})

DatatypeValidator.register_type("vector",
{
	get_type = function(value)
		return vector.check(value)
	end,

	conversions = 
	{
		["string"] = function(value)
			return tostring(value)
		end
	}
})

DatatypeValidator.register_type("objectref", 
{
	get_type = function(value)
		return value.get_pos and value:get_pos() ~= nil
	end,

	conversions = 
	{
		["string"] = function(value)
			if value:is_player() then
				return value:get_player_name()
			elseif value.nametag then
				return value.nametag
			end
		end,

		["vector"] = function(value)
			return value:get_pos()
		end,

		["player_name"] = function(value)
			if value:is_player() then
				return value:get_player_name()
			end
		end,

		["player"] = function(value)
			if value:is_player() then
				return value
			end
		end,
	},
})

DatatypeValidator.register_type("player_name",
{
	get_type = function(value)
		return type(value) == "string" and minetest.get_player_by_name(value) ~= nil
	end,

	conversions = 
	{
		["string"] = function(value)
			return value
		end,

		["objectref"] = function(value)
			return minetest.get_player_by_name(value)
		end,
	},
})

DatatypeValidator.register_type("player",
{
	get_type = function(value)
		return value.get_pos and value:get_pos() and value:is_player()
	end,

	conversions = 
	{
		["string"] = function(value)
			return value:get_player_name()
		end,

		["player_name"] = function(value)
			return value:get_player_name()
		end,

		["objectref"] = function(value)
			return value
		end,

		["vector"] = function(value)
			return value:get_pos()
		end,
	},
})

return DatatypeValidator
