-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

lush = {}

--- creating a compatibility require() 
local old_require = require
local loaded_modules = {}
local current_directory = minetest.get_modpath("lush_core") .. "/src/"
local redirect_modules =
{
	["helpers"] = "minetest/helpers.lua",
	["command"] = "minetest/command.lua",
	["env"] = "minetest/env.lua",
	["language_constructs.command_invocation"] = "minetest/language_constructs/command_invocation.lua",
	["language_constructs.vector_expression"] = "minetest/language_constructs/vector_expression.lua",
	["language_constructs.inline_lua"] = "minetest/language_constructs/inline_lua.lua",
	["datatype_validator"] = "minetest/datatype_validator.lua",
	["selector"] = "minetest/selector.lua",

}

require = function(module)
	local path = current_directory .. string.gsub(module, "%.", "/") .. ".lua"

	-- if isnt loaded
	if not loaded_modules[module] then
		if redirect_modules[module] then
			local redirect_value = redirect_modules[module]
			redirect_modules[module] = nil -- to prevent recursion
			loaded_modules[module] = dofile(current_directory .. redirect_value)
		else
			loaded_modules[module] = dofile(path)
		end
	end

	return loaded_modules[module]
end

local function get_setting(key, default) 
	return minetest.settings:get("lush_" .. key) or default
end

lush.Storage = require("minetest.storage")

lush.Env = require("env")
lush.Command, lush.Registered_Commands = require("command")
lush.ShellContext = require("shell_context")
lush.ParserContext = require("parser_context")
lush.DatatypeValidator = require("datatype_validator")
lush.helpers = require("helpers")
lush.Selector = require("selector")
lush.Parser = require("parser")

lush.global_env = lush.Env:new({parameters = lush.Storage.load("global_parameters", {})})
lush.spawnpoints = lush.Storage.load("spawnpoints", {})

local function save_data()
	lush.Storage.save("spawnpoints", lush.spawnpoints)
	-- lush.Storage.save("player_environments", lush.player_environments)
	
	local player_parameters = {}
	for player_name, player_env in pairs(lush.player_environments) do
		player_parameters[player_name] = {}
		for param, value in pairs(lush.player_environments[player_name].parameters) do
			player_parameters[player_name][param] = value
		end
	end

	lush.Storage.save("player_parameters", player_parameters)

	local global_parameters = {}
	for param, value in pairs(lush.global_env.parameters) do
		global_parameters[param] = value
	end

	lush.Storage.save("global_parameters", global_parameters)
end

local timer = 0
minetest.register_globalstep(function(dtime)
    timer = timer + dtime
    if timer > get_setting("save_interval", 100) then
        timer = 0
		save_data()
    end
end)

minetest.register_on_shutdown(
function()
	save_data()
end)

minetest.register_on_mods_loaded(
function()
    minetest.register_on_respawnplayer(
	function(player)
        local name = player:get_player_name()
        if lush.spawnpoints[name] then
            player:set_pos(lush.spawnpoints[name])
            return true
        end
    end)
end)
