-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod ShellContext <br>
-- this module defines the ShellContext class <br>
-- a structure that acts like a protocol to fallicitate dataflow from <br>
-- different commands or the shell itself

--- @todo stdin, stdout and stderr should be their own custom class that does type checking whether all arguments are of same type
-- and other useful thingies

local Env = require("env")
local TypedList = require("typed_list")

-- @type ShellContext

local ShellContext =
{
	instance_of = "ShellContext"
}

ShellContext.__index = ShellContext

--- constructor
-- @tab t definition table with matching keys as the class
-- @treturn Env the newly created Env instance
function ShellContext:new(t)
	if not t then
		t = {}
	end

	-- {{{ type checking
	if type(t) ~= "table" then error("t should be of type table. instead of " .. type(t)) end
	if t.stdin and t.stdin.instance_of ~= "TypedList" then error("t.stdin should be an instance of TypedList. instead of " .. t.stdin.instance_of) end
	if t.stdout and t.stdout.instance_of ~= "TypedList" then error("t.stdout should be an instance of TypedList. instead of " .. t.stdout.instance_of) end
	if t.stderr and type(t.stderr) ~= "string" then error("t.stderr should be of type string. instead of " .. type(t.stderr)) end
	if t.env and t.env.instance_of ~= "Env" then error("env should be an instance of Env. instead of " .. self.env.instance_of) end
	if t.exit_status and type(t.exit_status) ~= "number" then error("t.exit_status should be of type number. instead of " .. type(t.exit_status)) end
	-- }}}

	self = setmetatable(
	{
		--- @TypedList stdin
		stdin = t.stdin or TypedList:new(),

		--- @TypedList stdout
		stdout = t.stdout or TypedList:new(),

		--- @string stderr
		stderr = t.stderr or "",

		--- @number exit_status a value specifying exit status, equal to 0 when sucess, non zero when failure
		exit_status = t.exit_status or 0,

		--- @Env env container for all the shell's parameters
		env = t.env or Env:new()

	}, ShellContext)
	self.__index = self

	return self
end

--- sets the exit status and stderr
-- @string message message to be written to stderr
-- @treturn nil
function ShellContext:error(message)
	self.exit_status = 1
	self.stdin:clear()
	self.stdout:clear()
	self.stderr = message
end

return ShellContext
