-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests/busted_definitions")

local inspect = require("inspect")

describe("Command class: ",
function()
	local Command
	local ShellContext = require("shell_context")
	local example_command
	local shell_context_instance
	local s

	setup(
	function()
		Command = require("command")
	end)

	before_each(
	function()
		example_command = Command:new(
		{
			name = "example",
			allowed_options = {["option"] = false, ["option-argument"] = true},
			callback = function(context, args, options)

			end
		})

		s = spy.on(example_command, "callback")
	end)

	after_each(
	function()
		example_command:unregister()
		s = nil
	end)

	it("newly created instance is in the Registered_Commands table",
	function()
		local is_registered = Command.is_name_registered("example")

		assert.is_true(is_registered)
	end)

	it("call() calls the callback function",
	function()
		shell_context_instance = ShellContext:new()

		example_command:call(shell_context_instance, {"1", "2", "3"})

		assert.spy(s).was_called()
	end)

	it("call() splits the arguments into options correctly",
	function()
		shell_context_instance = ShellContext:new()

		example_command:call(shell_context_instance, {"1", "--option", "2", "3", "--option-argument", "argument"})

		assert.spy(s).was_called_with(match._, match._, match.is_all_of(match.has_property("option"), match.has_property("option-argument")))
	end)

	it("call() extracts arguments for options that require it",
	function()
		shell_context_instance = ShellContext:new()

		example_command:call(shell_context_instance, {"1", "2", "3", "--option-argument", "argument"})

		assert.spy(s).was_called_with(match._, match._, match.has_property_equaling("option-argument", "argument"))
	end)

	it("call() errors when an option that expects an argument dosent get it",
	function()
		shell_context_instance = ShellContext:new()

		assert.errors(function()
			example_command:call(shell_context_instance, {"1", "2", "3", "--option-argument"})
		end)

	end)

	it("call() errors given an invalid option",
	function()
		shell_context_instance = ShellContext:new()

		assert.errors(function()
			example_command:call(shell_context_instance, {"1", "2", "3", "--invalid_option"})
		end)
	end)

	describe("all members are correctly set by new()",
	function()
		local definition_table

		before_each(
		function()
			definition_table =
			{
				name = "command",
				allowed_options = {["h"] = true},
				callback = function(context, args, options) end
			}
			example_command:unregister()
			example_command = Command:new(definition_table)
		end)

		it("name",
		function()
			assert.equals(definition_table.name, example_command.name)
		end)

		it("allowed_options",
		function()
			assert.equals(definition_table.allowed_options, example_command.allowed_options)
		end)

		it("callback",
		function()
			assert.equals(definition_table.callback, example_command.callback)
		end)
	end)
end)
