-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod Env <br>
-- container type which acts as a key/value storage for shell parameters
local inspect = require("inspect")

--- @type Env

local Env =
{
	instance_of = "Env",
}

Env.__index = Env

--- constructor
-- @treturn Env Env instance
function Env:new(t)
	if t == nil then
		t = {}
	end

	self = setmetatable(
	{
		--- @tab parameters list containing all the defined parameters <br>
		--  key - string name of the parameter
		--  value - value of the parameter
		parameters = t.parameters or {},

		--- @tab read_only a set of keys for parameters that are read only
		read_only = t.read_only or {},

		--- @tab volatile_parameters same as Env.parameters, but isnt persistent over server restarts
		volatile_parameters = t.volatile_parameters or {},

		--- @Env an enviroment to inherit parameters from if they are undefined in local scope
		inherited_env = t.inherited_env or {}
	}, Env)

	return self
end

--- overwrite the default __newindex() function to creates new members inside the parameters table
-- @string k key
-- @string v value
function Env:__newindex(k, v)
	-- {{{ type checking
	if type(k) ~= "string" then error("k should be of type string. instead of " .. type(k)) end
	-- }}}

	rawset(self.parameters, k, v)
end

--- gets the value of a parameter with the given key
--	it will look for the parameter in the following order: <br>
--	first it checks whether the parameter is read only in any of the inherited environments, if so, returns the parameter value in the top most environment
--	otherwise it checks whether it exists as a volatile parameter, if not then it looks whether its a normal parameter, and if not then it tries to inherit it from other environments
--	@string key
--	@return value
function Env:get(key)
	-- {{{ type checking
	if key == nil then return nil end
	if type(key) ~= "string" then error("key should be of type string. instead of " .. type(key)) end
	-- }}}

	local output

	if self.inherited_env.instance_of == "Env" and self:is_param_readonly(key) then
		output = self.inherited_env:get(key)
	end
	output =  output or self.volatile_parameters[key] or self.parameters[key] or (self.inherited_env.instance_of == "Env" and self.inherited_env:get(key)) or nil

	if type(output) == "function" then
		output = output(self)
	end

	return output

end

-- sets a parameter with the given key and value,
-- optionally can also provide flags read_only and is_volatile
-- @string key
-- @param value
-- @bool read_only whether the value can be modified with Env:set
-- @bool is_volatile whether the parameter will be persistent over server reloads 
function Env:set(key, value, read_only, is_volatile)
	-- {{{ type checking
	if type(key) ~= "string" then error("key should be of type string. instead of " .. type(key)) end
	if self:is_param_readonly(key) then error("trying to set a read only parameter") end
	-- }}}

	if read_only then
		self.read_only[key] = true
	end

	if is_volatile then
		self.volatile_parameters[key] = value
	else
		self.parameters[key] = value
	end
end

--- removes a parameter with the given key
--	@string key
--	@treturn nil
function Env:unset(key)
	-- {{{ type checking
	if type(key) ~= "string" then error("key should be of type string. instead of " .. type(key)) end
	if self.read_only[key] then error("trying to unset a read only parameter") end
	-- }}}

	self.parameters[key] = nil
end

--- checks whether the parameter with the given key is read only
--	@string key
--	@treturn bool
function Env:is_param_readonly(key)
	if self.read_only[key] then
		return true
	elseif self.inherited_env.instance_of == "Env" then
		return self.inherited_env:is_param_readonly(key)
	end
end

--- "inherits" from a different environment
--	this means that when trying to access an undefined parameter, it will check whether
--	the parameter exists in that environment, and if so, returns it
--	@Env env
--	@treturn nil
function Env:inherit(env)
	-- {{{ type checking
	if env.instance_of ~= "Env" then error("env should be an instance of Env. instead of " .. env.instance_of) end
	-- }}}

	self.inherited_env = env
end

return Env
