-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_TableExpression
--	describes a vector

local inspect = require("inspect")
local helpers = require("helpers")
local Expression
local StringExpression

local TableExpression =
{
	instance_of = "LanguageConstruct_TableExpression",
}

TableExpression.__index = TableExpression

function TableExpression:new(t)
	t = t or {}
	self = setmetatable(
	{
		key_container = t.key_container or {},
		value_container = t.value_container or {}
	}, TableExpression)

	return self
end

function TableExpression:evaluate(context)
	-- {{{ type checking
	if context.instance_of ~= "ShellContext" then error("context should be an instance of ShellContext. instead of " .. context.instance_of) end
	-- }}}

	local output = {}

	for i = 1, #self.key_container do
		output[self.key_container[i]:evaluate(context)] = self.value_container[i]:evaluate(context)
	end

	return output
end

-- @fixme add infinite ranges support
-- the state comments dont include white spaces because it would be too verbose to be useful
function TableExpression:parse(parser_context)
	-- {{{ type checking
	if parser_context.instance_of ~= "ParserContext" then error("parser_context should be an instance of ParserContext. instead of " .. parser_context.instance_of) end
	-- }}}

	Expression = Expression or require("language_constructs.expression")
	StringExpression = StringExpression or require("language_constructs.string_expression")

	local char = parser_context:consume()
	local key
	local value

	if char ~= "[" then
		error([[table is expected to start with ", instead of ]] .. char)
	end

	while true do
		parser_context:skip_until(helpers.white_chars_set, true)

		key = StringExpression:new()
		key:parse(parser_context,
		{
			["\t"] = true,
			[" "] = true,
			["="] = true,
		})

		-- state: <name>*

		char = parser_context:peek()

		if char ~= "=" then
			parser_context:skip_until(helpers.white_chars_set, true)
		end


		parser_context:advance()
		parser_context:skip_until(helpers.white_chars_set, true)

		-- state: <name>=*

		print("parsing value")
		value = Expression:new()
		value:parse(parser_context,
		{
			["\t"] = true,
			[" "] = true,
			[","] = true,
			["]"] = true,
		})
		print("stopped parsing value")

		-- state: <name>=<value>*

		parser_context:skip_until(helpers.white_chars_set, true)
		char = parser_context:consume()


		table.insert(self.key_container, key)
		table.insert(self.value_container, value)

		if char == "]" then
			break
		elseif char ~= "," then
			error("unexpected character " .. char)
		end
	end

end

return TableExpression
