-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local Range = require("range")

local Selector = {}

Selector.SelectorPresets =
{
	["e"] =
	{
		selector_type = "object",
		sort = "nearest",
	},

	["s"] =
	{
		selector_type = "object",
		sort = "nearest",
		type = "player",
		limit = 1,
	},

	["p"] =
	{
		selector_type = "object",
		sort = "nearest",
		type = "player",
		limit = 1,
		["!radius"] = Range:new(
		{
			start = 0,
			finish = 0
		})
	},

	["r"] =
	{
		selector_type = "object",
		sort = "random",
		type = "player",
		limit = 1,
	},

	["a"] =
	{
		selector_type = "object",
		type = "player"
	},

	["c"] =
	{

	}

}

Selector.Matchers =
{
	["radius"] = function(object, param, context)
		if param.instance_of ~= "Range" then
			error("radius is an invalid range")
		end

		return param:contains(vector.distance(object:get_pos(), context.env:get("pos")))
	end,

	["name"] = function(object, param, context)
		if type(param) ~= "string" then
			error("type must be a string")
		end

		if object:is_player() then
			return param == obj:get_player_name()
		else
			local tmp = object:get_luaentity()
			if tmp then
				return param == tmp._nametag or param == tmp.nametag
			else
				return false
			end
		end
	end,

	["type"] = function(object, param, context)
		if type(param) ~= "string" then
			error("type must be a string")
		end

		if param == "player" then
			return object:is_player()
		elseif param == "luaentity" then
			return object.get_luaentity ~= nil and object:get_luaentity() ~= nil
		end
	end
}

Selector.SortingMethods =
{
	["nearest"] = function(a, b, context, params)
		-- minetest.debug("sorting", a, dump(b))
		return vector.distance(a:get_pos(), context.env:get("pos")) < vector.distance(b:get_pos(), context.env:get("pos"))
	end,
	["furthest"] = function(a, b, context, params)
		return vector.distance(a:get_pos(), context.env:get("pos")) > vector.distance(b:get_pos(), context.env:get("pos"))
	end
}

function Selector.expand(preset, explicit_parameters, context)
	local output = {}
	local selector_params = {}

	-- dictionary where the key is a selector param name
	-- and the value is true if the parameter starts with !
	-- aka matches only elements that return false
	local inverted_params = {}

	-- there is a slight inefficiency here
	-- TableExpression:evaluate() loops over every element of the table and sets the elements to the output table
	-- and now we take that output table and loop over every element of it
	-- instead it would be much better to loop over the table single time by passing a table reference to
	-- TableExpression:evaluate() as an argument
	-- but i dont want to couple the code and include non standard arguments to any evaluate() function

	local tmp
	for k, v in pairs(Selector.SelectorPresets[preset]) do
		if string.sub(k, 1, 1) == "!" then
			tmp = string.sub(k, 2, -1)
			inverted_params[tmp] = true
			selector_params[tmp] = v
		else
			selector_params[k] = v
		end
	end

	if explicit_parameters then
		for k, v in pairs(explicit_parameters) do
			if string.sub(k, 1, 1) == "!" then
				tmp = string.sub(k, 2, -1)
				inverted_params[tmp] = true
				selector_params[tmp] = v
			else
				selector_params[k] = v
			end
		end
	end

	if selector_params.selector_type == "object" then
		-- im using a seperate list instead of "output"
		-- because there might hundreds of objects on the server
		-- and if i were to directly assign them to "output"
		-- i would need to remove a majority of them
		-- there would be a lot of unnecesary shifting of elements back
		-- so i would prefer to avoid that
		-- (objects are only references, it dosent waste that much memory)
		local rough_objects = {}
		local valid

		if selector_params.radius and selector_params.radius.instance_of == "Range" and not inverted_params["radius"] then
			-- preforming a little optimization
			-- only objects in the radius are considered
			-- note that this is only a rough list of entities
			-- objects that are below the radius's lower bound and ones that are right on the finish when the rang
			-- are put in the table too
			rough_objects = minetest.get_objects_inside_radius(context.env:get("pos"), selector_params.radius.finish)
		else
			for _, luaentity in pairs(minetest.luaentities) do
				if luaentity.object:get_pos() then
					table.insert(rough_objects, luaentity.object)
				end
			end
		end

		-- adding all the players
		for _, player in pairs(minetest.get_connected_players()) do
			if player:get_pos() then
				table.insert(rough_objects, player)
			end
		end



		for _, object in pairs(rough_objects) do
			if object:get_pos() then
				valid = true
				for selector_param_key, selector_param_value in pairs(selector_params) do
					if Selector.Matchers[selector_param_key] and Selector.Matchers[selector_param_key](object, selector_param_value, context) == (inverted_params[selector_param_key] ~= nil) then
						valid = false
					end
				end

				if valid then
					table.insert(output, object)
				end
			end
		end

		if selector_params["sort"] == "random" then
			table.shuffle(output)
		else
			if Selector.SortingMethods[selector_params["sort"]] then
				table.sort(output, function(a, b)
					-- minetest.debug("iteration", dump(a), dump(b))
					return Selector.SortingMethods[selector_params["sort"]](a, b, context, selector_params)
				end)
			else
				error("invalid sorting method")
			end
		end

		table.sort(output, function(a, b)
			return vector.distance(a:get_pos(), context.env:get("pos")) < vector.distance(b:get_pos(), context.env:get("pos"))
		end)

		-- now im going to enforce the limit of objects
		-- im setting to rough_objects to avoid the
		-- element shifting overhead again
		rough_objects = output
		output = {}

		if type(selector_params["limit"]) == "number" then
			for i = 1, selector_params["limit"] do
				table.insert(output, rough_objects[i])
			end
		else
			if selector_params["limit"] == nil then
				output = rough_objects
			else
				error("limit must be set to a number")
			end
		end

	else
		error("invalid selector_type")
	end

	return output
end

return Selector
