-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_StringExpression
--	Acts as an abstraction for DoubleQuoteString, SingleQuoteString and UnquotedString language constructs

local DoubleQuoteString
local SingleQuoteString
local UnquotedString

local StringExpression =
{
	instance_of = "LanguageConstruct_StringExpression",
}

StringExpression.__index = StringExpression

function StringExpression:new()
	self = setmetatable(
	{
		expression = nil
	}, StringExpression)

	return self
end

function StringExpression:evaluate(context)
	-- {{{ type checking
	if  not
		(
			self.expression.instance_of == "LanguageConstruct_SingleQuoteString" or
			self.expression.instance_of == "LanguageConstruct_DoubleQuoteString" or
			self.expression.instance_of == "LanguageConstruct_UnquotedString"
		)
		then error("expression should be an instance of DoubleQuoteString or SingleQuoteString") end
	-- }}}

	return self.expression:evaluate(context)
end

function StringExpression:parse(parser_context, terminator_set)
	terminator_set = terminator_set or {}

	local first_char = parser_context:peek()
	UnquotedString = UnquotedString or require("language_constructs.unquoted_string")

	if first_char == "\"" then
		DoubleQuoteString = DoubleQuoteString or require("language_constructs.double_quote_string")
		self.expression = DoubleQuoteString:new()
		self.expression:parse(parser_context)
	elseif first_char == "'" then
		SingleQuoteString = SingleQuoteString or require("language_constructs.single_quote_string")
		self.expression = SingleQuoteString:new()
		self.expression:parse(parser_context)
	else
		self.expression = UnquotedString:new()
		self.expression:parse(parser_context, terminator_set)
	end

end

return StringExpression
