-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @submodule Command

local Command, RegisteredCommands = require("command")
local Parser = require("parser")
local helpers = require("helpers")
local ShellContext = require("shell_context")
local ParserContext = require("parser_context")
local Expression = require("language_constructs.expression")
local DatatypeValidator = require("datatype_validator")

local inspect = require("inspect")

local old_constructor = Command.new
Command.new = function(self, t)
	-- minetest.log(inspect(t))
	-- minetest.log(inspect(Command.is_name_registered(t.name)))
	self = old_constructor(self, t)

	-- {{{ type checking
	if t.privs and type(t.privs) ~= "table" then error("t.privs should be of type table. instead of " .. type(t.privs)) end
	if t.params and type(t.params) ~= "string" then error("t.params should be of type string. instead of " .. type(t.params)) end
	if t.description and type(t.description) ~= "string" then error("t.description should be of type string. instead of " .. type(t.description)) end
	-- }}}

	self.privs = t.privs or {}
	self.params = t.params or ""
	self.description = t.description or {}
	self.mod_origin = minetest.get_current_modname()

	if self.arguments and self.params == "" then
		self.params = DatatypeValidator.generate_param_hint(self.arguments)
	end

	if minetest.registered_chatcommands[self.name] then
		minetest.log("info", string.format("%s overriden by lush", self.name))
		minetest.unregister_chatcommand(self.name)
	end

	self:register_compatibility_command()

	return self
end

local old_unregister = Command.unregister
Command.unregister = function(self)
	old_unregister(self)
	minetest.unregister_chatcommand(self.name)
end

local old_call = Command.call
Command.call = function(self, shell_context_instance, args)
	local has, missing = minetest.check_player_privs("singleplayer", self.privs)

	if has then
		old_call(self, shell_context_instance, args)
	else
		error("player is missing privilages: " .. dump(missing))
	end
end

Command.register_compatibility_command = function(self)
	minetest.register_chatcommand(self.name,
	{
		privs = self.privs,
		params = self.params,
		description = self.description,
		func =
		function(name, params)
			--- @todo make a proper compatibility layer that uses the parser
			local shell_context_instance = ShellContext:new({env = lush.player_environments[name]})
			local parser_context_instance = ParserContext:new({text = params})
			local args = {}
			local arg

			while true do
				parser_context_instance:skip_until(helpers.white_chars_set, true)

				if parser_context_instance:is_EOF() then
					break
				end

				arg = Expression:new()
				arg:parse(parser_context_instance, helpers.white_chars_set)

				table.insert(args, arg:evaluate(shell_context_instance))
			end

			local success, message = pcall(self.call, self, shell_context_instance, args)

			-- print(dump(shell_context_instance))
			if success then
				shell_context_instance.stdout:cast_to("string")
				return true, table.concat(shell_context_instance.stdout.container, "\n")
			else
				return false, message
			end
		end
	})
end

minetest.register_on_chat_message(
function(name, message)
	if string.sub(message, 1, 1) == "!" then
		Parser.run_shell(string.sub(message, 2), ShellContext:new({env = lush.player_environments[name]}), true)
		return true
	end
end)

minetest.register_on_mods_loaded(function()
	for command, def in pairs(RegisteredCommands) do
		def:register_compatibility_command()
		minetest.registered_chatcommands[command].mod_origin = def.mod_origin
	end
end)

return Command, RegisteredCommands
