-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

minetest.unregister_chatcommand("mods")

minetest.register_privilege("lush_heal",
{
	description = "whether the player can use the heal command",
	give_to_singleplayer = false
})

minetest.register_privilege("lush_damage",
{
	description = "whether the player can use the damage/kill command",
	give_to_singleplayer = false
})

minetest.register_privilege("invisible",
{
	description = "whether the player can use the invis command",
	give_to_singleplayer = false
})

minetest.register_privilege("clear",
{
	description = "whether the player can use the clear command",
	give_to_singleplayer = false
})

minetest.register_privilege("fix",
{
	description = "whether the player can use the fix_items command",
	give_to_singleplayer = false
})

local max = tonumber(minetest.settings:get("lush_max_lag")) or 2.0

local lag = 0
local last = minetest.get_us_time()
minetest.register_globalstep(function()
		if lag <= 0 then 
			return
		end

		local exp = last + lag * 1000000
		while minetest.get_us_time() < exp do 
			-- looping until time passes
		end

		last = exp
	end)

lush.Command:new(
{
	name = "lag",
	arguments = 
	{
		{
			name = "time_per_globalstep",
			type = "time",
			single = true
		}
	},
	callback = function(context, args, options)
		if args["time_per_globalstep"] > max then
			error("time per globalstep can not exceed " .. tostring(max) .. " seconds")
		end

		lag = args["time_per_globalstep"]
	end,
	description = "lags out the server so each global step takes a specific amount of seconds",
	privs = {server = true}
})

lush.Command:new(
{
	name = "days",
	options = {},
	arguments = {},
	callback = function(context, args, options)
		context.stdout:push(minetest.get_day_count())
	end,
	privs = {},
	description = "prints the amounts of days passed",
})

lush.Command:new(
{
	name = "set_node",
	arguments = 
	{
		{
			name = "pos",
			type = "vector",
			single = false
		},
		{
			name = "node",
			type = "node_name",
			single = true
		},
	},
	callback = function(context, args, options)
		for i, pos in args["pos"]:iterator() do
			minetest.set_node(pos, {name = args["node"]})
		end
	end,
	privs = {give = true},
	description = "sets a node",
})

local storage = minetest.get_mod_storage()

local time_update_job

local function time_update_func(time)
	minetest.set_timeofday(time)
	time_update_job = minetest.after(1, time_update_func, time)
end

if storage:contains("lush_permament_time") and storage:get_float("lush_permament_time") ~= -1 then
	time_update_job = minetest.after(1, time_update_func, storage:get_float("lush_permament_time"))
end

lush.Command:new(
{
	name = "time",
	options = {["permament"] = false},
	arguments =
	{
		{
			name = "time",
			type = {"time", "string"},
			single = true
		}
	},
	callback = function(context, args, options)
		local time_of_day = 0

		if type(args["time"]) == "string" then
			if args["time"] == "midnight" then
				time_of_day = 0
			elseif args["time"] == "night" then
				time_of_day = 0.8
			elseif args["time"] == "sunset" then
				time_of_day = 0.75
			elseif args["time"] == "noon" then
				time_of_day = 0.5
			elseif args["time"] == "day" then
				time_of_day = 0.3
			elseif args["time"] == "sunrise" then
				time_of_day = 0.2
			else
				error(string.format("%s isnt a recgonized time", args["time"]))
			end
		else
			if args["time"] >= 86400 or args["time"] < 0 then
				error("time must be between 0 seconds and 24 hours")
			end

			time_of_day = args["time"] / 86400
		end

		minetest.set_timeofday(time_of_day)

		if options["permament"] then
			time_update_job = minetest.after(1, time_update_func, time_of_day)
			storage:set_float("lush_permament_time", time_of_day)
		else
			time_update_job:cancel()
			storage:set_float("lush_permament_time", -1)
		end
	end,
	privs = {settime = true},
	description = "sets time",
})

lush.Command:new(
{
	name = "lunch",
	options = {},
	arguments =
	{
		{
			name = "objects",
			type = "objectref",
			single = false
		},
		{
			name = "vector",
			type = "vector",
			single = true
		}
	},
	callback = function(context, args, options)
		for i, v in args["objects"]:iterator() do
			v:add_velocity(args["vector"])
		end
	end,
	privs =
	{
		lush_damage = true
	},
	description = "adds velocity to entities",
})

lush.Command:new(
{
	name = "teleport",
	options = {},
	arguments =
	{
		{
			name = "objects",
			type = "objectref",
		},
		{
			rule_type = "terminal",
			value = "to",
			optional = true,
		},
		{
			name = "pos",
			type = "vector",
			single = true
		}
	},
	callback = function(context, args, options)
		local lm = 31007 -- equals MAX_MAP_GENERATION_LIMIT in C++
		local pos = args["pos"]
		if pos.x < -lm or pos.x > lm or pos.y < -lm or pos.y > lm
				or pos.z < -lm or pos.z > lm then
			error("position out of map's bounds")
		end

		if (context.env:get("name") ~= args["objects"]:get(1):get_player_name() or #args["objects"] > 1) and not minetest.check_player_privs(context.env:get("name"), "bring") then
			error("you need the 'bring' privilege to teleport anything other than yourself")
		end

		for i, v in args["objects"]:iterator() do
			if v:get_pos() and not v:get_attach() then
				v:set_pos(args["pos"])
			end
		end

	end,
	privs = {teleport = true},
	description = "teleports entities to position",
})

lush.Command:new(
{
	name = "kill",
	options = {},
	arguments =
	{
		{
			name = "objects",
			type = "objectref",
		},
	},
	callback = function(context, args, options)
		for i, v in args["objects"]:iterator() do
			lush.helpers.deal_damage(v, v:get_hp(), nil, true)
		end

	end,
	privs = {lush_damage = true},
	description = "kills entities",
})

lush.Command:new(
{
	name = "damage",
	options =
	{
		["type"] = true,
	},
	arguments =
	{
		{
			name = "targets",
			type = "objectref",
			single = false,
		},
		{
			name = "amount",
			type = "number",
			single = true,
		},
	},
	callback = function(context, args, options)
		local damage_type = "set_hp" or options["type"]

		if type(damage_type) ~= "string" then
			error("damage type should be a string")
		end

		for i, v in args["targets"]:iterator() do
			lush.helpers.deal_damage(v, args["amount"], nil, true)
		end

	end,
	privs = {lush_damage = true},
	description = "deals damage to entities",
})

lush.Command:new(
{
	name = "spawnpoint",
	options = {},
	arguments =
	{
		{
			name = "names",
			type = "player_name",
			single = false,
		},
		{
			name = "pos",
			type = "vector",
			single = true,
		},
	},
	callback = function(context, args, options)
		for _, v in args["names"]:iterator() do
			lush.spawnpoints[v] = args["pos"]
		end

		lush.Storage.save("spawnpoints", lush.spawnpoints)
	end,
	privs = {},
	description = "sets the respawn point for the player",
})

lush.Command:new(
{
	name = "clear_spawnpoint",
	options = {},
	arguments =
	{
		{
			name = "names",
			type = "player_names",
			single = false,
		},
	},
	callback = function(context, args, options)
		for _, v in args["names"]:iterator() do
			lush.spawnpoints[v] = nil
		end

		lush.Storage.save("spawnpoints", lush.spawnpoints)
	end,
	privs = {},
	description = "removes the previously set spawnpoint",
})

lush.Command:new(
{
	name = "heal",
	options = {},
	arguments =
	{
		{
			name = "targets",
			type = "objectref",
			single = false,
		},
		{
			name = "amount",
			type = "number",
			single = true,
		},
	},
	callback = function(context, args, options)
		for i, v in args["targets"]:iterator() do
			lush.helpers.deal_damage(v, -args["amount"], nil, false)
		end
	end,
	privs =
	{
		lush_heal = true
	},
	description = "heals entities by an amount",
})

lush.Command:new(
{
	name = "list",
	options = {},
	arguments = {},
	callback = function(context, args, options)
		for _, player in ipairs(minetest.get_connected_players()) do
			context.stdout:push(player:get_player_name())
		end
	end,
	privs =
	{
		debug = true
	},
	description = "lists online players",
})

lush.Command:new(
{
	name = "seed",
	options = {},
	arguments = {},
	callback = function(context, args, options)
		context.stdout:push(minetest.get_mapgen_setting("seed"))
	end,
	privs =
	{
		debug = true
	},
	description = "prints the world's seed",
})

lush.Command:new(
{
	name = "lsmods",
	options = {},
	arguments = {},
	callback = function(context, args, options)
		for i, v in ipairs(minetest.get_modnames()) do
			context.stdout:push(v)
		end
	end,
	privs =
	{
		debug = true
	},
	description = "prints active mods to STDOUT",
})

-- credit to SkyBuilder1717's Essentials mod
-- from whom this implementation is partly ripped
lush.Command:new(
{
	name = "invis",
	arguments =
	{
		{
			type = "player",
			name = "players",
			single = false
		},
		{
			type = "string",
			name = "is_enabled",
			single = true
		}
	},
	callback = function(context, args, options)
		local prop

		if args["is_enabled"] == "on" then
			prop = {
				visual_size = {x = 0, y = 0, z = 0},
				is_visible = false,
				nametag_color = {r=255,g=255,b=255,a=255},
				pointable = false,
				makes_footstep_sound = false,
				show_on_minimap = false,
			}
		elseif args["is_enabled"] == "off" then
			prop = {
				visual_size = {x = 1, y = 1, z = 1},
				is_visible = true,
				nametag_color = {r=255,g=255,b=255,a=0},
				pointable = true,
				makes_footstep_sound = true,
				show_on_minimap = true,
			}
		else
			error("invalid argument: '" .. tostring(args["is_enabled"]) .. "', should be either on or off")
		end

		for i, v in args["players"]:iterator() do
			v:set_properties(prop)

			if args["is_enabled"] == "on" then
				v:get_meta():set_int("invisible", 1)
			elseif args["is_enabled"] == "off" then
				v:get_meta():set_int("invisible", 0)
			end
		end
	end,
	privs =
	{
		invisible = true
	},
	description = "makes the entities invisible",
})

lush.Command:new(
{
	name = "give",
	options =
	{
		["wear"] = true,
		["amount"] = true
	},
	arguments =
	{
		{
			type = "player",
			name = "players",
			single = false
		},
		{
			type = "string",
			name = "item_name",
			single = true
		},
		{
			type = "table",
			name = "meta",
			single = true
		}
	},
	callback = function(context, args, options)
		local stack = ItemStack({name = args["item_name"]})

		local meta = stack:get_meta()
		meta:from_table({fields = args["meta"]})

		if options["wear"] then
			stack:set_wear(options["wear"])
		end

		if options["amount"] then
			stack:set_count(options["amount"])
		end

		local inv
		for i, object in args["players"]:iterator() do
			inv = object:get_inventory()

			inv:add_item("main", stack)
		end

	end,
	privs = {give = true},
	description = "gives items",
})

lush.Command:new(
{
	name = "pulverize",
	options =
	{
	},
	arguments =
	{
		{
			name = "items",
			type = "inventory_item_list",
			single = true
		},
	},
	callback = function(context, args, options)
		local lists = args["items"]:get_lists()

		for listname, stacks in pairs(lists) do
			for _, stack in pairs(stacks) do
				stack:clear()
			end
		end

		args["items"]:set_lists(lists)
	end,
	privs = {clear = true},
	description = "clears items",
})

lush.Command:new(
{
	name = "fix_items",
	arguments =
	{
		{
			name = "items",
			type = "inventory_item_list",
			single = true
		},
	},
	callback = function(context, args, options)
		local lists = args["items"]:get_lists()

		for listname, stacks in pairs(lists) do
			for _, stack in pairs(stacks) do
				stack:set_wear(0)
			end
		end

		args["items"]:set_lists(lists)
	end,
	privs = {fix = true},
	description = "fixes items",
})

lush.Command:new(
{
	name = "nodeinfo",
	options =
	{
		["human"] = false
	},
	arguments =
	{
		{
			rule_type = "terminal",
			name = "type",
			value =
			{
				["metadata"] = true,
				["inventory"] = true,
				["groups"] = true,
				["name"] = true,
				["param1"] = true,
				["param2"] = true,
			}
		},
		{
			name = "pos",
			type = "vector",
			single = true
		}
	},
	callback = function(context, args, options)

		if args["type"] == "metadata" then
			local meta = minetest.get_meta(args["pos"]):to_table()

			for k, v in pairs(meta.fields) do
				context.stdout:push(tostring(k) .. "=" .. tostring(v))
			end
		elseif args["type"] == "inventory" then
			local inv = minetest.get_inventory({type = "node", pos = args["pos"]})

			if inv then
				if options["human"] then
					-- print in tree like format
					-- the code looks funny because the last element must start with
					-- an "└─ " even if the last item in the inventory is empty
					-- this is a very clever solution to this problem
					for list_name, list in pairs(inv:get_lists()) do
						context.stdout:push(list_name)

						local previous_item
						for i, item in pairs(list) do

							if not item:is_empty() then
								if previous_item then
									context.stdout:push("├─ " .. previous_item:to_string())
								end
								previous_item = item
							end
						end

						if previous_item then
							context.stdout:push("└─ " .. previous_item:to_string())
						end
					end
				else
					for list_name, list in pairs(inv:get_lists()) do
						context.stdout:push("List " .. list_name)
						for _, v in pairs(list) do
							context.stdout:push(v:to_string())
						end
						context.stdout:push("EndInventoryList")
					end
				end
			end
		elseif args["type"] == "groups" then
			local nodedef = minetest.registered_nodes[minetest.get_node(args["pos"]).name]

			for k, v in pairs(nodedef.groups) do
				context.stdout:push(tostring(k) .. "=" .. tostring(v))
			end
		elseif args["type"] == "name" then
			context.stdout:push(minetest.get_node(args["pos"]).name)
		elseif args["type"] == "param1" then
			context.stdout:push(minetest.get_node(args["pos"]).param1)
		elseif args["type"] == "param2" then
			context.stdout:push(minetest.get_node(args["pos"]).param2)
		end
	end,
	privs =
	{
		debug = true
	},
	description = "prints information about the node at a given position",
})
