-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local storage = minetest.get_mod_storage()

local blacklisted_commands = minetest.deserialize(storage:get("blacklisted_commands") or "") or {}

--  key-value pair
--  key: command name
--  value: a table with fields:
--    type: command type, either "lush" or "minetest"
--    privs: the new required privileges
--    original_privs: the new required privileges
local overriden_command_privs = minetest.deserialize(storage:get("overriden_command_privs") or "") or {}

local function prune_blacklisted_commands()
	for command, _ in pairs(blacklisted_commands) do
		if minetest.registered_chatcommands[command] then
			minetest.registered_chatcommands[command] = nil
		end

		if lush.Command.is_name_registered(command) then
			lush.Command.get_command_by_name(command):unregister()
		end
	end
end

local function command_exists(command_name)
	if minetest.registered_chatcommands[command_name] then
		return true
	end

	if lush.Command.is_name_registered(command_name) then
		return true
	end

	return false
end

minetest.register_privilege("command_managment",
{
	description = "Can use command managment commands",
})

lush.Command:new(
{
	name = "blacklist_command",
	description = "blacklists a commands from being used",
	privs = {command_managment = true},
	arguments = 
	{
		{
			rule_type = "block",
			name = "command",
			type = "string",
			required = true
		}
	},
	callback = function(context, args, options)
		local invalid_commands = {}
		for _, command in pairs(args["command"]) do
			if not command_exists(command) then
				table.insert(invalid_commands, command)
			end

			blacklisted_commands[command] = true
		end

		prune_blacklisted_commands()
		storage:set_string("blacklisted_commands", minetest.serialize(blacklisted_commands))

		if #invalid_commands > 0 then
			error(string.format("cant blacklist commands because they dont exist: %s", table.concat(invalid_commands, ", ")))
		end
	end
})

lush.Command:new(
{
	name = "whitelist_command",
	description = "whitelist a command that was previously blacklisted, requires server restart to take effect",
	privs = {command_managment = true},
	arguments = 
	{
		{
			rule_type = "block",
			name = "command",
			type = "string",
			required = true
		}
	},
	callback = function(context, args, options)
		for _, command in pairs(args["command"]) do
			blacklisted_commands[command] = nil
		end

		storage:set_string("blacklisted_commands", minetest.serialize(blacklisted_commands))
	end
})

local function override_command_privs(command_name)
	local override = overriden_command_privs[command_name]
	if override.type == "lush" then
		lush.RegisteredCommands[command_name].privs = override.privs
	else
		minetest.registered_chatcommands[command_name].privs = override.privs
	end
end

lush.Command:new(
{
	name = "override_command_privs",
	description = "changes the required privileges of a command",
	privs = {command_managment = true},
	arguments = 
	{
		{
			name = "command",
			type = "string",
			single = true
		},
		{
			rule_type = "terminal",
			name = "action",
			value = {["add"] = true, ["remove"] = true, ["set"] = true}
		},
		{
			rule_type = "block",
			name = "privs",
			type = "string",
			required = true
		}
	},
	callback = function(context, args, options)
		if not command_exists(args["command"]) then
			error("command dosent exist: " .. args["command"])
		end

		local old_privs
		local delta_privs = {}
		for i, v in pairs(args["privs"]) do
			delta_privs[v] = true
		end

		overriden_command_privs[args["command"]] = {}

		if lush.Command.is_name_registered(args["command"]) then
			old_privs = lush.RegisteredCommands[args["command"]].privs
			overriden_command_privs[args["command"]].type = "lush"
		else
			old_privs = minetest.registered_chatcommands[args["command"]].privs
			overriden_command_privs[args["command"]].type = "minetest"
		end

		overriden_command_privs[args["command"]].original_privs = overriden_command_privs[args["command"]].original_privs or old_privs

		if args["action"] == "add" then
			overriden_command_privs[args["command"]].privs = lush.helpers.concat_sets(old_privs, delta_privs)
		elseif args["action"] == "remove" then
			for i, v in pairs(delta_privs) do
				old_privs[i] = nil
			end

			overriden_command_privs[args["command"]].privs = old_privs
		elseif args["action"] == "set" then
			overriden_command_privs[args["command"]].privs = delta_privs
		end

		override_command_privs(args["command"])
		storage:set_string("overriden_command_privs", minetest.serialize(overriden_command_privs))
	end
})

lush.Command:new(
{
	name = "restore_command_privs",
	description = "restores the default required privileges of a command",
	privs = {command_managment = true},
	arguments = 
	{
		{
			name = "command",
			type = "string",
			single = true
		}
	},
	callback = function(context, args, options)
		if not command_exists(args["command"]) then
			error("command dosent exist: " .. args["command"])
		end


		local override = overriden_command_privs[args["command"]]
		if overriden_command_privs[args["command"]].type == "lush" then
			lush.RegisteredCommands[args["command"]].privs = overriden_command_privs[args["command"]].original_privs
		else
			minetest.registered_chatcommands[args["command"]].privs = overriden_command_privs[args["command"]].original_privs
		end

		overriden_command_privs[args["command"]] = nil
		storage:set_string("overriden_command_privs", minetest.serialize(overriden_command_privs))
	end
})

lush.Command:new(
{
	name = "ls_command_overrides",
	description = "lists commands that got their privs overriden",
	privs = {command_managment = true},
	arguments = {},
	callback = function(context, args, options)
		local priv_string
		local first
		for command_name, override in pairs(overriden_command_privs) do
			priv_string = ""
			first = true

			-- checking for added privileges
			for priv, _ in pairs(override.privs) do
				if not override.original_privs[priv] then
					if not first then
						priv_string = priv_string .. ", "
					end

					priv_string = priv_string .. minetest.colorize("lime", priv)
					first = false
				end
			end

			-- checking for removed privileges
			for priv, _ in pairs(override.original_privs) do
				if not override.privs[priv] then
					if not first then
						priv_string = priv_string .. ", "
					end

					priv_string = priv_string .. minetest.colorize("red", priv)
					first = false
				end
			end

			context.stdout:push(command_name .. ": " .. priv_string)
		end
	end
})

minetest.register_on_mods_loaded(function()
	for command_name, _ in pairs(overriden_command_privs) do
		override_command_privs(command_name)
	end
end)
