-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @module helpers this is a module with helper functions

local helpers = {}

local ParserContext
local StringExpression

-- {{{ defining various sets and lists

helpers.control_operators_list =
{
	"|",
	"&&",
	"||",
	";",
	"\n"
}

helpers.valid_identifier_chars_set =
{
    ["A"] = true,
	["a"] = true,
	["B"] = true,
	["b"] = true,
	["C"] = true,
	["c"] = true,
	["D"] = true,
	["d"] = true,
	["E"] = true,
	["e"] = true,
	["F"] = true,
	["f"] = true,
	["G"] = true,
	["g"] = true,
	["H"] = true,
	["h"] = true,
	["I"] = true,
	["i"] = true,
	["J"] = true,
	["j"] = true,
	["K"] = true,
	["k"] = true,
	["L"] = true,
	["l"] = true,
	["M"] = true,
	["m"] = true,
	["N"] = true,
	["n"] = true,
	["O"] = true,
	["o"] = true,
	["P"] = true,
	["p"] = true,
	["Q"] = true,
	["q"] = true,
	["R"] = true,
	["r"] = true,
	["S"] = true,
	["s"] = true,
	["T"] = true,
	["t"] = true,
	["U"] = true,
	["u"] = true,
	["V"] = true,
	["v"] = true,
	["W"] = true,
	["w"] = true,
	["X"] = true,
	["x"] = true,
	["Y"] = true,
	["y"] = true,
	["Z"] = true,
	["z"] = true,
	["_"] = true,
}

helpers.numeric_chars_set =
{
	["1"] = true,
	["2"] = true,
	["3"] = true,
	["4"] = true,
	["5"] = true,
	["6"] = true,
	["7"] = true,
	["8"] = true,
	["9"] = true,
	["0"] = true,

}

helpers.white_chars_set =
{
	[" "] = true,
	["\t"] = true,
}

-- }}}

--- returns whether str follows the format of an option and the option name
-- ie starts with one hyphen and one characters
-- or two hyphens and arbitrary amount of characters
-- @string str the string to checking
-- treturn bool whether is a valid option
function helpers.is_option(str)
	-- {{{ type checking
	if type(str) ~= "string" then error("str should be of type string. instead of " .. type(str)) end
	-- }}}

	local _, _, name = string.find(str, "^%-([%a_])$")
	if name == nil then
		_, _, name = string.find(str, "^%-%-([%a_][%a%d_-]+)$")
		if name == nil then
			return false, nil
		else
			return true, name
		end
	else
		return true, name
	end
end

--- returns whether v is a vector
-- @vector v the vector to check
-- treturn bool whether is a vector
function helpers.is_vector(v)
	return type(v) == "table"
	and (#v == 3 and type(v[1]) == "number" and type(v[2]) == "number" and type(v[3]) == "number")
end

--- returns whether the given string is a singular character or empty
--	@string char the string to check
--	@treturn bool whether is a single character or empty
function helpers.is_single_character(char)
	-- {{{ type checking
	if type(char) ~= "string" then error("char should be of type string. instead of " .. type(char)) end
	-- }}}

	return #char <= 1
end

--- @function helpers.is_valid_identifier_char
--	@string char the character to check
--	@bool is_leading flag that decides whether numeric characters are allowed
--	@treturn bool whether is a valid character for identifier names
--	@see helpers.is_valid_identifier_name
function helpers.is_valid_identifier_char(char, is_leading)
	-- {{{ type checking
		if type(char) ~= "string" then error("char should be of type string. instead of " .. type(char)) end
		if not helpers.is_single_character(char) then error("char should be a single character") end
	-- }}}

	return char ~= "" and (helpers.valid_identifier_chars_set[char]
	or (not is_leading and helpers.numeric_chars_set[char]))
end

--- @function helpers.is_valid_identifier_name
--	@string text the name to check
--	treturns bool whether is a valid name for an identifier
--	@see helpers.is_valid_identifier_char
function helpers.is_valid_identifier_name(text)
	-- {{{ type checking
	if type(text) ~= "string" then error("text should be of type string. instead of " .. type(text)) end
	-- }}}

	if not helpers.is_valid_identifier_char(string.sub(text, 1, 1), true) then return false end
	for i = 1, #text do
		if not helpers.is_valid_identifier_char(string.sub(text, i, i)) then return false end
	end

	return true
end

--- creates a valid StringExpression instance from a string, is pretty wasteful over doing it the normal way
--	@string text the string to be turned into a string expression
--	@treturn StringExpression the newly created instance
function helpers.create_string_expression(text)
	ParserContext = require("parser_context")
	StringExpression = require("language_constructs.string_expression")

	local output_instance = StringExpression:new()

	local parser_context_instance = ParserContext:new({text = text})

	output_instance:parse(parser_context_instance)

	return output_instance
end

--- splits the given string into a table
--	@string text the string to split
--	@string seperator a single character that seperates the elements
--	@bool allow_empty_elements whether empty elements should be created when given a string with consecutive seperators
--	@treturn tab a list of strings
function helpers.string_split(text, seperator, allow_empty_elements)
	-- {{{ type checking
		if type(text) ~= "string" then error("text should be of type string. instead of " .. type(text)) end
		if type(seperator) ~= "string" then error("seperator should be of type string. instead of " .. type(seperator)) end
		if #seperator ~= 1 then error("seperator should be a single character. instead of " .. #seperator) end
	-- }}}

	if allow_empty_elements == nil then
		allow_empty_elements = true
	end

	local output_table = {}
	local element_start = 1
	local element_end = 1
	local char

	repeat
		char = string.sub(text, element_end, element_end)
		if char == seperator then
			-- if consecutive seperators and not allowing empty elements
			if allow_empty_elements or element_start ~= element_end then
				table.insert(output_table, string.sub(text, element_start, element_end - 1))
			end
			element_start = element_end + 1
		end

		element_end = element_end + 1
	until char == ""

	if allow_empty_elements or element_start + 1 ~= element_end then
		table.insert(output_table, string.sub(text, element_start, element_end - 1))
	end

	return output_table
end

--- crates a new list with the elements of the provided lists
--	@tab t1 the second list
--	@tab t2 the first list
--	@treturn tab returns a new list with the elements of both lists
function helpers.concat_lists(t1, t2)
	local output_table = {unpack(t1)}
	for _, v in pairs(t2) do
		table.insert(output_table, v)
	end

	return output_table
end

--- crates a new set with elements from both provided sets
--	@tab t2 the first set
--	@tab t1 the second set
--	@treturn tab returns a new set with the elements of both lists
function helpers.concat_sets(...)
	local output_table = {}
	local args = {...}
	for _, t in pairs(args) do
		for k, v in pairs(t) do
			output_table[k] = v
		end
	end

	return output_table
end

local time_units = 
{
	["s"] = 1,
	["m"] = 60,
	["h"] = 3600,
	["D"] = 86400,
	["W"] = 604800,
	["M"] = 2629746,
	["Y"] = 31556952,
}

--- parses a time string
--	@string str string to parse
--	@treturn number|nil returns a number of seconds, or nil if isnt a parse string
function helpers.parse_time(str)
	ParserContext = require("parser_context")
	local time = 0
	local parser_context = ParserContext:new({text = str})

	parser_context:skip_until(helpers.white_chars_set, true)

	local unit
	local number
	local number_start_idx = 1
	while not parser_context:is_EOF() do
		number_start_idx = parser_context.character_index
		parser_context:skip_until(
		{
			["0"] = true,
			["1"] = true,
			["2"] = true,
			["3"] = true,
			["4"] = true,
			["5"] = true,
			["6"] = true,
			["7"] = true,
			["8"] = true,
			["9"] = true,
			["."] = true,
		}, true)

		number = tonumber(str:sub(number_start_idx, parser_context.character_index - 1))
		
		if number == nil or unit and unit <= time_units[parser_context:peek()] then
			return nil
		end

		unit = time_units[parser_context:consume()]
		if unit then
			time = time + number * unit
		end

		parser_context:skip_until(helpers.white_chars_set, true)
	end

	return time
end

local time_unit_amounts_ordered = 
{
	31556952,
	2629746,
	604800,
	86400,
	3600,
	60,
	1,
}

local time_units_ordered = 
{
	"Y",
	"M",
	"W",
	"D",
	"h",
	"m",
	"s",
}

--- converts a number of seconds to a time string
--	@number time number of seconds
--	@treturn string time string
function helpers.time_to_string(time)
	local time_str = ""

	local tmp
	for i, unit_amount in pairs(time_unit_amounts_ordered) do
		tmp = math.floor(time / unit_amount)
		
		if tmp ~= 0 then
			time_str = time_str .. tostring(tmp) .. time_units_ordered[i]
			time = time - tmp * unit_amount
		end
	end

	return time_str
end

return helpers
