-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod ParserContext <br>
-- to be honest, i have no expierience in making handmade parsers and i thought this was a good learning expierience <br>
-- (thats why i didnt use a generator) <br>
-- so dont be mad if some things arent understandable <br>
-- here is the overall plan how it works: <br>
-- the ParserContext class is a wrapper around the input text that provides peek() and consume() methods <br>
-- when parsing the parser attempts to recognize what language construct is next and calls parse() method of the appopriate LanguageConstruct derivative classes <br>
-- that does all the parsing and returns control to the parser in a different location of the parsed text <br>

local inspect = require("inspect")

--- @type ParserContext
local ParserContext =
{
	instance_of = "ParserContext",
}

ParserContext.__index = ParserContext

--- constructor
-- @tab t table with matching keys as the class, with initialization values for the members
-- @treturn ParserContext the newly created Command instance
function ParserContext:new(t)
	if t == nil then
		t = {}
	end

	-- {{{ type checking
	if type(t) ~= "table" then error("t should be of type table. instead of " .. type(t)) end
	if t.character_index and type(t.character_index) ~= "number" then error("t.character_index should be of type number. instead of " .. type(t.character_index)) end
	if t.text and type(t.text) ~= "string" then error("t.text should be of type string. instead of " .. type(t.text)) end
	-- }}}

	self = setmetatable(
	{
		--- @number[opt=1] value specifying the index of next character to be read
		character_index = t.character_index or 1,
		--- @string[opt=""] text to be parser
		text = t.text or ""

	}, ParserContext)

	return self
end

--- returns the next nth character from the current position
-- @number[opt=1] offset the amount of characters to advance
-- @treturn string a single character
-- @see ParserContext:consume
function ParserContext:peek(offset)
	offset = offset or 1

	local offset_index = self.character_index + offset - 1
	return string.sub(self.text, offset_index, offset_index)
end

--- returns the next nth character from the current position and updates the current character index to that position
-- @number[opt=1] offset the amount of characters to advance
-- @treturn string a single character
-- @see ParserContext:peek, ParserContext:rewind
function ParserContext:consume(offset)
	offset = offset or 1

	self.character_index = self.character_index + offset

	-- print("consumed character: ", string.sub(self.text, self.character_index - 1, self.character_index - 1), "now on index: ", self.character_index)
	return string.sub(self.text, self.character_index - 1, self.character_index - 1)
end

--- returns to the previous nth position
-- @number[opt=1] offset the amount of characters to rewind
-- @treturn nil
-- @see ParserContext:consume
function ParserContext:rewind(offset)
	offset = offset or 1

	self.character_index = self.character_index - offset
end

function ParserContext:advance(offset)
	offset = offset or 1

	self.character_index = self.character_index + offset
end

--- consumes characters until it finds one that is in the terminator set (or the end of string),
--	if the current character is in the terminator set, does nothing
--	@tab terminator_set a set of characters
--	@bool inverse if true then it will skip to the first character that isnt in the terminator set
--	@treturn nil
function ParserContext:skip_until(terminator_set, inverse)
	-- {{{ type checking
	if type(terminator_set) ~= "table" then error("terminator_set should be of type table. instead of " .. type(terminator_set)) end
	if inverse and type(inverse) ~= "boolean" then error("inverse should be of type boolean. instead of " .. type(inverse)) end
	-- }}}

	local offset = 1
	while terminator_set[self:peek(offset)] == inverse and self.character_index + offset <= #self.text + 1 do
		offset = offset + 1
	end

	if offset ~= 1 then
		self.character_index = self.character_index + offset - 1
	end
end

--- tries to match any of the specified terminators, starting from the current position and returns the longest match <br>
--	note: if your terminators are single characters, then its recommended that you use ParserContext:skip_until instead
--	@tab a list of terminators
--	@bool consume whether to consume the matched terminator
--	@return the frst return value a bool which says whether any terminator matched, if so then the second value is the terminator that matched, if no terminator matched, then the second value is nil
function ParserContext:match(terminator_list, consume)
	if terminator_list == nil then
		terminator_list = {}
	end

	-- {{{ type checking
	if type(terminator_list) ~= "table" then error("terminator_list should be of type table. instead of " .. type(terminator_list)) end
	-- }}}

	-- the code is kinda messy but its the only way i could think of that also covers all edge cases

	local index_offest = 0
	local longest_matching_terminator
	local possible_matches_ammount = #terminator_list
	local char

	-- a set of indexes in the terminator_list table
	-- for terminators that dont match
	local nonmatching_terminator_index_set = {}

	while possible_matches_ammount > 0 do
		index_offest = index_offest + 1
		char = self:peek(index_offest)

		-- print("")
		-- print("ITERATION")
		-- print("inspecting character", char)
		-- print("index", index_offest)
		-- print("terminator list", inspect(terminator_list))

		for i, terminator in ipairs(terminator_list) do
			-- if the terminator is still a possible match
			if not nonmatching_terminator_index_set[i] then
				-- if terminator's character dosent match, remove that terminator
				if string.sub(terminator, index_offest, index_offest) ~= char then
					nonmatching_terminator_index_set[i] = true
					possible_matches_ammount = possible_matches_ammount - 1

				-- if the terminator character does match and is the last character of the terminator
				elseif #terminator == index_offest or char == "" then
					longest_matching_terminator = terminator

					possible_matches_ammount = possible_matches_ammount - 1
					nonmatching_terminator_index_set[i] = true
				end
			end
		end

		-- print("mismatched terminators", inspect(nonmatching_terminator_index_set))
		-- print("best match", longest_matching_terminator)
		-- print("possible matches", possible_matches_ammount)
	end

	if longest_matching_terminator then
		if consume then
			self.character_index = self.character_index + #longest_matching_terminator
		end

		return true, longest_matching_terminator
	else
		return false, nil
	end
end

--- returns true if is at the end of the string
function ParserContext:is_EOF()
	return self.character_index > #self.text
end

--- method called when an error occurs while parsing
-- @string message the error message
-- treturn nil
function ParserContext:error(message)
	print(tostring(self.character_index) .. ": " .. tostring(message))
end

return ParserContext
