-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 


-- @fix this file requires the minetest namespace

local ShellContext = require("shell_context")
local Shell = require("language_constructs.shell")
local ParserContext = require("parser_context")
local DataTypeValidator = require("datatype_validator")
local helpers = require("helpers")

local Parser =
{
	instance_of = "Parser"
}

--- executes a shell
--	@string code the code to execute
--	@ShellContext initial_context the shell_context the shell will start with
--	@bool share_env whether the shell will share the environment of the passed ShellContext, this means that any parameter assignments will be persistent
function Parser.run_shell(code, initial_context, share_env)
	local t1 = minetest.get_us_time()
	if initial_context == nil then
		initial_context = {}
	end

	local parser_context = ParserContext:new({text = code})
	local shell_context = ShellContext:new(initial_context)
	local shell = Shell:new()
	local success
	local message
	local parse_time
	local eval_time

	minetest.log("action", tostring(shell_context.env:get("name")) .. " invoked a lush shell: \n" .. tostring(code))
	minetest.chat_send_player(tostring(shell_context.env:get("name")), "invoked lush:")

	success, message = pcall(shell.parse, shell, parser_context, {}, {})

	if not success then
		message = message or "no error message provided"
		minetest.log("encountered an error while parsing: " .. tostring(message))
	end

	parse_time = minetest.get_us_time() - t1
	t1 = minetest.get_us_time()

	success, message = pcall(shell.evaluate, shell, shell_context, share_env)
	eval_time = minetest.get_us_time() - t1

	if shell_context.exit_status == 0 and success then
		shell_context.stdout:cast_to("string")
		for i, v in shell_context.stdout:iterator() do
			minetest.chat_send_player(shell_context.env:get("name"), v)
		end
	else
		message = message or shell_context.stderr or "no error message provided"
		minetest.chat_send_player(shell_context.env:get("name"), "encountered an error while evaluating: " .. message)
		minetest.log("encountered an error while evaluating: " .. message)
	end

	minetest.log("action", "exit code: " .. tostring(shell_context.exit_status)
	.. "\nparsing time: " .. tostring(parse_time / 1000) .. "ms"
	.. "\nevaluation time: " .. tostring(eval_time / 1000) .. "ms")

end

return Parser
