-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests/busted_definitions")
local inspect = require("inspect")

describe("ParameterExpansion class: ",
function()
	local ShellContext = require("shell_context")
	local ParameterExpansion = require("language_constructs.parameter_expansion")
	local ParserContext = require("parser_context")
	local helpers = require("helpers")
	local shell_context_instance
	local parameter_expansion_instance
	local parser_context_instance

	before_each(
	function()
		shell_context_instance = ShellContext:new()
		parameter_expansion_instance = ParameterExpansion:new()
		parser_context_instance = ParserContext:new()

		shell_context_instance.env:set("parameter", "value")

	end)

	after_each(
	function()
		shell_context_instance.env:unset("undefined_parameter")
	end)

	it("constructs without errors",
	function()
		assert.has_no.errors(function() ParameterExpansion:new(shell_context_instance) end)
	end)

	describe("evaluates correctly ",
	function()

		it("without modifiers",
		function()
			parameter_expansion_instance.parameter_name = "parameter"

			assert.equals("value", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("without modifiers when given an undefined parameter",
		function()
			parameter_expansion_instance.parameter_name = "undefined_parameter"

			assert.equals("", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"fallback\" modifier when given an undefined parameter",
		function()
			parameter_expansion_instance.parameter_name = "undefined_parameter"
			parameter_expansion_instance.modifier = parameter_expansion_instance.valid_modifiers.fallback
			parameter_expansion_instance.modifier_args[1] = helpers.create_string_expression("fallback")

			assert.equals("fallback", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"fallback\" modifier when given a defined parameter",
		function()
			parameter_expansion_instance.parameter_name = "parameter"
			parameter_expansion_instance.modifier = parameter_expansion_instance.valid_modifiers.fallback
			parameter_expansion_instance.modifier_args[1] = helpers.create_string_expression("fallback")

			assert.equals("value", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"fallback_set\" modifier when given an undefined parameter",
		function()
			parameter_expansion_instance.parameter_name = "undefined_parameter"
			parameter_expansion_instance.modifier = parameter_expansion_instance.valid_modifiers.fallback_set
			parameter_expansion_instance.modifier_args[1] = helpers.create_string_expression("fallback")

			assert.equals("fallback", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"fallback_set\" modifier when given a defined parameter",
		function()
			parameter_expansion_instance.parameter_name = "parameter"
			parameter_expansion_instance.modifier = parameter_expansion_instance.valid_modifiers.fallback_set
			parameter_expansion_instance.modifier_args[1] = helpers.create_string_expression("fallback")

			assert.equals("value", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"guard\" modifier when given an undefined parameter",
		function()
			parameter_expansion_instance.parameter_name = "undefined_parameter"
			parameter_expansion_instance.modifier = parameter_expansion_instance.valid_modifiers.guard
			parameter_expansion_instance.modifier_args[1] = helpers.create_string_expression("error message")

			local s = spy.on(shell_context_instance, "error")

			assert.equals("", parameter_expansion_instance:evaluate(shell_context_instance))
			assert.spy(s).called_with(match._, "error message")
		end)

		it("with \"guard\" modifier when given a defined parameter",
		function()
			parameter_expansion_instance.parameter_name = "parameter"
			parameter_expansion_instance.modifier = parameter_expansion_instance.valid_modifiers.guard
			parameter_expansion_instance.modifier_args[1] = helpers.create_string_expression("error message")

			assert.equals("value", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"offset\" modifier when given a defined parameter",
		function()
			parameter_expansion_instance.parameter_name = "parameter"
			parameter_expansion_instance.modifier = parameter_expansion_instance.valid_modifiers.offset
			parameter_expansion_instance.modifier_args[1] = helpers.create_string_expression("4")

			assert.equals("ue", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"offset\" modifier when offset is larger than the length of the parameter",
		function()
			parameter_expansion_instance.parameter_name = "parameter"
			parameter_expansion_instance.modifier = parameter_expansion_instance.valid_modifiers.offset
			parameter_expansion_instance.modifier_args[1] = helpers.create_string_expression("999")

			assert.equals("", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"offset\" modifier when given an undefined parameter",
		function()
			parameter_expansion_instance.parameter_name = "undefined_parameter"
			parameter_expansion_instance.modifier = parameter_expansion_instance.valid_modifiers.offset
			parameter_expansion_instance.modifier_args[1] = helpers.create_string_expression("5")

			assert.equals("", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"length\" modifier when given a defined parameter",
		function()
			parameter_expansion_instance.parameter_name = "parameter"
			parameter_expansion_instance.additional_modifier = parameter_expansion_instance.valid_additional_modifiers.length

			assert.equals(5, parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"length\" modifier when given an undefined parameter",
		function()
			parameter_expansion_instance.parameter_name = "undefined_parameter"
			parameter_expansion_instance.additional_modifier = parameter_expansion_instance.valid_additional_modifiers.length

			assert.equals(0, parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"indirection\" modifier when given a defined parameter",
		function()
			shell_context_instance.env:set("value", "value2") --creating a new parameter with the name of the "parameter" parameter's value

			parameter_expansion_instance.parameter_name = "parameter"
			parameter_expansion_instance.additional_modifier = parameter_expansion_instance.valid_additional_modifiers.indirection

			assert.equals("value2", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"indirection\" modifier when given an undefined parameter",
		function()
			shell_context_instance.env:set("value", "value2") --creating a new parameter with the name of the "parameter" parameter's value

			parameter_expansion_instance.parameter_name = "undefined_parameter"
			parameter_expansion_instance.additional_modifier = parameter_expansion_instance.valid_additional_modifiers.indirection

			assert.equals("", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"substitution\" modifier when given a defined parameter",
		function()
			parameter_expansion_instance.parameter_name = "parameter"
			parameter_expansion_instance.modifier = parameter_expansion_instance.valid_modifiers.substitution
			parameter_expansion_instance.modifier_args[1] = helpers.create_string_expression("val")
			parameter_expansion_instance.modifier_args[2] = helpers.create_string_expression("replacement ")

			assert.equals("replacement ue", parameter_expansion_instance:evaluate(shell_context_instance))
		end)

		it("with \"substitution\" modifier when given an undefined parameter",
		function()
			parameter_expansion_instance.parameter_name = "undefined_parameter"
			parameter_expansion_instance.modifier = parameter_expansion_instance.valid_modifiers.substitution
			parameter_expansion_instance.modifier_args[1] = helpers.create_string_expression("val")
			parameter_expansion_instance.modifier_args[2] = helpers.create_string_expression("replacement ")

			assert.equals("", parameter_expansion_instance:evaluate(shell_context_instance))
		end)


	end)

	describe("parse(): ",
	function()

		it("parses correctly when given the \"$var\" type of expansion",
		function()
			parameter_expansion_instance = ParameterExpansion:new(shell_context_instance)
			parser_context_instance.text = "$parameter"
			parameter_expansion_instance:parse(parser_context_instance)

			assert.has_properties_equaling(
			{
				parameter_name = "parameter",
				modifier = parameter_expansion_instance.valid_modifiers.none
			}, parameter_expansion_instance)

			assert.same({}, parameter_expansion_instance.modifier_args)

			assert.equals(11, parser_context_instance.character_index)

		end)

		it("parses correctly when given the \"${var}\" type of expansion",
		function()
			parameter_expansion_instance = ParameterExpansion:new(shell_context_instance)
			parser_context_instance.text = "${parameter}"
			parameter_expansion_instance:parse(parser_context_instance)

			assert.has_properties_equaling(
			{
				parameter_name = "parameter",
				modifier = parameter_expansion_instance.valid_modifiers.none
			}, parameter_expansion_instance)

			assert.same({}, parameter_expansion_instance.modifier_args)

			assert.equals(13, parser_context_instance.character_index)

		end)

		it("parses correctly when given the \"${var:-word}\" type of expansion",
		function()
			parameter_expansion_instance = ParameterExpansion:new(shell_context_instance)
			parser_context_instance.text = "${parameter:-fallback text}"
			parameter_expansion_instance:parse(parser_context_instance)

			assert.has_properties_equaling(
			{
				parameter_name = "parameter",
				modifier = parameter_expansion_instance.valid_modifiers.fallback
			}, parameter_expansion_instance)

			assert.same("fallback text", parameter_expansion_instance.modifier_args[1]:evaluate(shell_context_instance))

			assert.equals(28, parser_context_instance.character_index)

		end)

		it("parses correctly when given the \"${var:=word}\" type of expansion",
		function()
			parameter_expansion_instance = ParameterExpansion:new(shell_context_instance)
			parser_context_instance.text = "${parameter:=fallback text}"
			parameter_expansion_instance:parse(parser_context_instance)

			assert.has_properties_equaling(
			{
				parameter_name = "parameter",
				modifier = parameter_expansion_instance.valid_modifiers.fallback_set
			}, parameter_expansion_instance)

			assert.same("fallback text", parameter_expansion_instance.modifier_args[1]:evaluate(shell_context_instance))

			assert.equals(28, parser_context_instance.character_index)

		end)

		it("parses correctly when given the \"${var:?word}\" type of expansion",
		function()
			parameter_expansion_instance = ParameterExpansion:new(shell_context_instance)
			parser_context_instance.text = "${parameter:?fallback text}"
			parameter_expansion_instance:parse(parser_context_instance)

			assert.has_properties_equaling(
			{
				parameter_name = "parameter",
				modifier = parameter_expansion_instance.valid_modifiers.guard
			}, parameter_expansion_instance)

			assert.same("fallback text", parameter_expansion_instance.modifier_args[1]:evaluate(shell_context_instance))

			assert.equals(28, parser_context_instance.character_index)

		end)

		it("parses correctly when given the \"${var:offset}\" type of expansion",
		function()
			parameter_expansion_instance = ParameterExpansion:new(shell_context_instance)
			parser_context_instance.text = "${parameter:5}"
			parameter_expansion_instance:parse(parser_context_instance)

			assert.has_properties_equaling(
			{
				parameter_name = "parameter",
				modifier = parameter_expansion_instance.valid_modifiers.offset
			}, parameter_expansion_instance)

			assert.same("5", parameter_expansion_instance.modifier_args[1]:evaluate(shell_context_instance))

			assert.equals(15, parser_context_instance.character_index)

		end)

		it("parses correctly when given the \"${var:offset:length}\" type of expansion",
		function()
			parameter_expansion_instance = ParameterExpansion:new(shell_context_instance)
			parser_context_instance.text = "${parameter:5:1}"
			parameter_expansion_instance:parse(parser_context_instance)

			assert.has_properties_equaling(
			{
				parameter_name = "parameter",
				modifier = parameter_expansion_instance.valid_modifiers.substring
			}, parameter_expansion_instance)

			assert.same("5", parameter_expansion_instance.modifier_args[1]:evaluate(shell_context_instance))
			assert.same("1", parameter_expansion_instance.modifier_args[2]:evaluate(shell_context_instance))

			assert.equals(17, parser_context_instance.character_index)

		end)

		it("parses correctly when given the \"${var/pattern/substitution}\" type of expansion",
		function()
			parameter_expansion_instance = ParameterExpansion:new(shell_context_instance)
			parser_context_instance.text = "${parameter/pattern/substitution}"
			parameter_expansion_instance:parse(parser_context_instance)

			assert.has_properties_equaling(
			{
				parameter_name = "parameter",
				modifier = parameter_expansion_instance.valid_modifiers.substitution
			}, parameter_expansion_instance)

			assert.same("pattern", parameter_expansion_instance.modifier_args[1]:evaluate(shell_context_instance))
			assert.same("substitution", parameter_expansion_instance.modifier_args[2]:evaluate(shell_context_instance))

			assert.equals(34, parser_context_instance.character_index)

		end)
	end)

	it("\"instance_of\" property is correctly set",
	function()
		assert.equals("LanguageConstruct_ParameterExpansion", parameter_expansion_instance.instance_of)
	end)
end)
