-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

describe("LanguageConstruct_RangeExpression",
function()
	local RangeExpression = require("language_constructs.range_expression")
	local ShellContext = require("shell_context")
	local ParserContext = require("parser_context")
	local shell_context_instance
	local parser_context_instance
	local range_expression_instance

	before_each(
	function()
		shell_context_instance = ShellContext:new()
		parser_context_instance = ParserContext:new()
		range_expression_instance = RangeExpression:new()
	end)

	describe("parses correctly: ",
	function()
		it("when given a simple inclusive range",
		function()
			parser_context_instance.text = "<1:2>"

			range_expression_instance:parse(parser_context_instance)

			assert.equals(1, range_expression_instance:evaluate(shell_context_instance).start)
			assert.equals(2, range_expression_instance:evaluate(shell_context_instance).finish)
			assert.equals(true, range_expression_instance:evaluate(shell_context_instance).start_inclusive)
			assert.equals(true, range_expression_instance:evaluate(shell_context_instance).finish_inclusive)
		end)

		it("when given a simple exclusive range",
		function()
			parser_context_instance.text = "(1:2)"

			range_expression_instance:parse(parser_context_instance)

			assert.equals(1, range_expression_instance:evaluate(shell_context_instance).start)
			assert.equals(2, range_expression_instance:evaluate(shell_context_instance).finish)
			assert.equals(false, range_expression_instance:evaluate(shell_context_instance).start_inclusive)
			assert.equals(false, range_expression_instance:evaluate(shell_context_instance).finish_inclusive)
		end)

		it("when given a mixed exclusive/inclusive range",
		function()
			parser_context_instance.text = "<1:2)"

			range_expression_instance:parse(parser_context_instance)

			assert.equals(1, range_expression_instance:evaluate(shell_context_instance).start)
			assert.equals(2, range_expression_instance:evaluate(shell_context_instance).finish)
			assert.equals(true, range_expression_instance:evaluate(shell_context_instance).start_inclusive)
			assert.equals(false, range_expression_instance:evaluate(shell_context_instance).finish_inclusive)
		end)

		it("when given a mixed exclusive/inclusive range #2",
		function()
			parser_context_instance.text = "(1:2>"

			range_expression_instance:parse(parser_context_instance)

			assert.equals(1, range_expression_instance:evaluate(shell_context_instance).start)
			assert.equals(2, range_expression_instance:evaluate(shell_context_instance).finish)
			assert.equals(false, range_expression_instance:evaluate(shell_context_instance).start_inclusive)
			assert.equals(true, range_expression_instance:evaluate(shell_context_instance).finish_inclusive)
		end)

		it("when given a range that goes to infinity",
		function()
			parser_context_instance.text = "(1:)"

			range_expression_instance:parse(parser_context_instance)

			assert.equals(1, range_expression_instance:evaluate(shell_context_instance).start)
			assert.equals(math.huge, range_expression_instance:evaluate(shell_context_instance).finish)
			assert.equals(false, range_expression_instance:evaluate(shell_context_instance).start_inclusive)
			assert.equals(false, range_expression_instance:evaluate(shell_context_instance).finish_inclusive)
		end)

		it("when given a range that goes to infinity into the negatives",
		function()
			parser_context_instance.text = "(:2)"

			range_expression_instance:parse(parser_context_instance)

			assert.equals(-math.huge, range_expression_instance:evaluate(shell_context_instance).start)
			assert.equals(2, range_expression_instance:evaluate(shell_context_instance).finish)
			assert.equals(false, range_expression_instance:evaluate(shell_context_instance).start_inclusive)
			assert.equals(false, range_expression_instance:evaluate(shell_context_instance).finish_inclusive)
		end)

		it("when given a range that has no beginning or end",
		function()
			parser_context_instance.text = "(:)"

			range_expression_instance:parse(parser_context_instance)

			assert.equals(-math.huge, range_expression_instance:evaluate(shell_context_instance).start)
			assert.equals(math.huge, range_expression_instance:evaluate(shell_context_instance).finish)
			assert.equals(false, range_expression_instance:evaluate(shell_context_instance).start_inclusive)
			assert.equals(false, range_expression_instance:evaluate(shell_context_instance).finish_inclusive)
		end)

		it("when given a range with excessive white spaces",
		function()
			parser_context_instance.text = "( 1 : 2 )"

			range_expression_instance:parse(parser_context_instance)

			assert.equals(1, range_expression_instance:evaluate(shell_context_instance).start)
			assert.equals(2, range_expression_instance:evaluate(shell_context_instance).finish)
			assert.equals(false, range_expression_instance:evaluate(shell_context_instance).start_inclusive)
			assert.equals(false, range_expression_instance:evaluate(shell_context_instance).finish_inclusive)
		end)
	end)
end)
