-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_TableExpression
--	describes a vector

local inspect = require("inspect")
local helpers = require("helpers")
local Expression
local StringExpression

local TableExpression =
{
	instance_of = "LanguageConstruct_TableExpression",
}

TableExpression.__index = TableExpression

function TableExpression:new(t)
	t = t or {}
	self = setmetatable(
	{
		-- key_container = t.key_container or {},
		-- value_container = t.value_container or {}
		container = t.container or {}
	}, TableExpression)

	return self
end

function TableExpression:evaluate(context)
	-- {{{ type checking
	if context.instance_of ~= "ShellContext" then error("context should be an instance of ShellContext. instead of " .. context.instance_of) end
	-- }}}

	local output = {}

	for key, value in pairs(self.container) do
		if type(key) ~= "number" then
			key = key:evaluate(context)

			if key == "instance_of" then
				error("'instance_of' is a restricted table key")
			end
		end

		output[key] = value:evaluate(context)
	end

	return output
end

-- @fixme add infinite ranges support
-- the state comments dont include white spaces because it would be too verbose to be useful
function TableExpression:parse(parser_context)
	-- {{{ type checking
	if parser_context.instance_of ~= "ParserContext" then error("parser_context should be an instance of ParserContext. instead of " .. parser_context.instance_of) end
	-- }}}

	Expression = Expression or require("language_constructs.expression")
	StringExpression = StringExpression or require("language_constructs.string_expression")

	local char = parser_context:consume()
	local key
	local value

	if char ~= "[" then
		error([[table is expected to start with ", instead of ]] .. char)
	end

	while true do
		parser_context:skip_until(helpers.white_chars_set, true)

		key = Expression:new()
		key:parse(parser_context,
		{
			["\t"] = true,
			[" "] = true,
			[","] = true,
			["="] = true,
			["]"] = true,
		})

		-- state: <name>*

		parser_context:skip_until(helpers.white_chars_set, true)

		char = parser_context:peek()

		if char == "=" then
			if key.expression.instance_of ~= "LanguageConstruct_DoubleQuoteString"
				and key.expression.instance_of ~= "LanguageConstruct_SingleQuoteString"
				and key.expression.instance_of ~= "LanguageConstruct_UnquotedString" 
				and key.expression.instance_of ~= "LanguageConstruct_StringExpression" then
				error("table key has to be a string")
			end

			parser_context:advance()
			parser_context:skip_until(helpers.white_chars_set, true)

			-- state: <name>=*

			value = Expression:new()
			value:parse(parser_context,
			{
				["\t"] = true,
				[" "] = true,
				[","] = true,
				["]"] = true,
			})

			-- state: <name>=<value>*

			parser_context:skip_until(helpers.white_chars_set, true)

			self.container[key] = value
		elseif char == "," or char == "]" then
			table.insert(self.container, key)
		else
			error("unexpected character: " .. char)
		end

		char = parser_context:consume()

		if char == "]" then
			break
		elseif char ~= "," then
			error("unexpected character " .. char)
		end
	end

end

return TableExpression
