-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local Range = require("range")
local InventoryItemList = require("minetest.inventory_item_list")

minetest.register_privilege("lush_get_inventories",
{
	description = "can get any inventory, including other player's",
	give_to_admin = false,
	give_to_singleplayer = false
})

local Selector = {}

Selector.SelectorPresets =
{
	["e"] =
	{
		selector_type = "object",
		sort = "nearest",
	},

	["s"] =
	{
		selector_type = "object",
		sort = "nearest",
		type = "player",
		limit = 1,
	},

	["p"] =
	{
		selector_type = "object",
		sort = "nearest",
		type = "player",
		limit = 1,
		["!radius"] = Range:new(
		{
			start = 0,
			finish = 0
		})
	},

	["n"] =
	{
		selector_type = "object",
		sort = "nearest",
		["!type"] = "player",
		limit = 1,
		["!radius"] = Range:new(
		{
			start = 0,
			finish = 0
		})
	},

	["r"] =
	{
		selector_type = "object",
		sort = "random",
		type = "player",
		limit = 1,
	},

	["a"] =
	{
		selector_type = "object",
		type = "player"
	},

	["c"] =
	{

	},

	["w"] =
	{
		selector_type = "item",
		wielded = true,
		location = 
		{
			type = "player",
			name = true
		}
	},

	["i"] =
	{
		selector_type = "item",
		list = "main",
		location = 
		{
			type = "player",
			name = true
		}
	},
}

Selector.Matchers =
{
	["object"] = 
	{
		["radius"] = function(object, param, context)
			-- only takes effect if player has lush_get_player_pos privilege
			-- lush_get_player_pos is defined in minetest/datatype_validator.lua
			if not minetest.check_player_privs(context.env:get("name"), "lush_get_object_pos") then
				return true
			end

			if param.instance_of ~= "Range" then
				error("radius is an invalid range")
			end

			return param:contains(vector.distance(object:get_pos(), context.env:get("pos")))
		end,

		["name"] = function(object, param, context)
			if type(param) ~= "string" then
				error("type must be a string")
			end

			if object:is_player() then
				return param == obj:get_player_name()
			else
				local tmp = object:get_luaentity()
				if tmp then
					return param == tmp._nametag or param == tmp.nametag
				else
					return false
				end
			end
		end,

		["type"] = function(object, param, context)
			if type(param) ~= "string" then
				error("type must be a string")
			end

			if param == "player" then
				return object:is_player()
			elseif param == "luaentity" then
				return object.get_luaentity ~= nil and object:get_luaentity() ~= nil and not object:is_player()
			end
		end
	},

	["item"] = 
	{
		["count"] = function(stack, param, context)
			if param.instance_of ~= "Range" then
				error("count argument must be a range")
			end

			return param:contains(stack:get_count())
		end,

		["max_stack"] = function(stack, param, context)
			if param.instance_of ~= "Range" then
				error("max_stack argument must be a range")
			end

			return param:contains(stack:get_stack_max())
		end,

		["wear"] = function(stack, param, context)
			if param.instance_of ~= "Range" then
				error("wear argument must be a range")
			end

			return param:contains(1 - (stack:get_wear() / 65535)) -- 65535 is the maximum wear
		end,

		["name"] = function(stack, param, context)
			if type(param) ~= "string" then
				error("name argument must be a string")
			end

			return string.find(stack:get_name(), param) ~= nil
		end,
	}
}

Selector.SortingMethods =
{
	["nearest"] = function(a, b, context, params)
		-- minetest.debug("sorting", a, dump(b))
		return vector.distance(a:get_pos(), context.env:get("pos")) < vector.distance(b:get_pos(), context.env:get("pos"))
	end,
	["furthest"] = function(a, b, context, params)
		return vector.distance(a:get_pos(), context.env:get("pos")) > vector.distance(b:get_pos(), context.env:get("pos"))
	end
}

function Selector.expand(preset, explicit_parameters, context)
	local output = {}
	local selector_params = {}

	-- dictionary where the key is a selector param name
	-- and the value is true if the parameter starts with !
	-- aka matches only elements that return false
	local inverted_params = {}

	-- there is a slight inefficiency here
	-- TableExpression:evaluate() loops over every element of the table and sets the elements to the output table
	-- and now we take that output table and loop over every element of it
	-- instead it would be much better to loop over the table single time by passing a table reference to
	-- TableExpression:evaluate() as an argument
	-- but i dont want to couple the code and include non standard arguments to any evaluate() function

	local tmp
	for k, v in pairs(Selector.SelectorPresets[preset]) do
		if string.sub(k, 1, 1) == "!" then
			tmp = string.sub(k, 2, -1)
			inverted_params[tmp] = true
			selector_params[tmp] = v
		else
			selector_params[k] = v
		end
	end

	if explicit_parameters then
		for k, v in pairs(explicit_parameters) do
			if string.sub(k, 1, 1) == "!" then
				tmp = string.sub(k, 2, -1)
				inverted_params[tmp] = true
				selector_params[tmp] = v
			else
				selector_params[k] = v
			end
		end
	end

	if selector_params.selector_type == "object" then
		-- im using a seperate list instead of "output"
		-- because there might hundreds of objects on the server
		-- and if i were to directly assign them to "output"
		-- i would need to remove a majority of them
		-- there would be a lot of unnecesary shifting of elements back
		-- so i would prefer to avoid that
		-- (objects are only references, it dosent waste that much memory)
		local rough_objects = {}
		local valid

		if selector_params.radius and selector_params.radius.instance_of == "Range" and not inverted_params["radius"] 
			and minetest.check_player_privs(context.env:get("name"), "lush_get_player_pos") then
			-- preforming a little optimization
			-- only objects in the radius are considered
			-- note that this is only a rough list of entities
			-- objects that are below the radius's lower bound and ones that are right on the finish 
			-- when the range isnt inclusive are put in the table too
			rough_objects = minetest.get_objects_inside_radius(context.env:get("pos"), selector_params.radius.finish)
		else
			for _, luaentity in pairs(minetest.luaentities) do
				if luaentity.object:get_pos() then
					table.insert(rough_objects, luaentity.object)
				end
			end
		end

		-- adding all the players
		for _, player in pairs(minetest.get_connected_players()) do
			if player:get_pos() then
				table.insert(rough_objects, player)
			end
		end



		for _, object in pairs(rough_objects) do
			if object:get_pos() then
				valid = true

				-- this loop could potentially be optimized
				-- its looping over all selector parameters, even those that arent matchers.
				-- creating a cache of ONLY parameters that are matchers would prevent this
				-- since its being run for every object, it might be worthwhile
				for selector_param_key, selector_param_value in pairs(selector_params) do
					if Selector.Matchers["object"][selector_param_key] and Selector.Matchers["object"][selector_param_key](object, selector_param_value, context) == (inverted_params[selector_param_key] ~= nil) then
						valid = false
					end
				end

				if valid then
					table.insert(output, object)
				end
			end
		end

		if selector_params["sort"] == "random" then
			table.shuffle(output)
		else
			if Selector.SortingMethods[selector_params["sort"]] then
				table.sort(output, function(a, b)
					-- minetest.debug("iteration", dump(a), dump(b))
					return Selector.SortingMethods[selector_params["sort"]](a, b, context, selector_params)
				end)
			elseif selector_params["sort"] then -- if a sorting method is speicified but not found
				error("invalid sorting method")
			end
		end

		table.sort(output, function(a, b)
			return vector.distance(a:get_pos(), context.env:get("pos")) < vector.distance(b:get_pos(), context.env:get("pos"))
		end)

		-- now im going to enforce the limit of objects
		-- im setting to rough_objects to avoid the
		-- element shifting overhead again
		rough_objects = output
		output = {}

		if type(selector_params["limit"]) == "number" then
			for i = 1, selector_params["limit"] do
				table.insert(output, rough_objects[i])
			end
		else
			if selector_params["limit"] == nil then
				output = rough_objects
			else
				error("limit must be set to a number")
			end
		end

	elseif selector_params.selector_type == "item" then

		-- hack
		if selector_params.location.type == "player" and selector_params.location.name == true then
			selector_params.location.name = context.env:get("name")
		elseif not minetest.check_player_privs(context.env:get("name"), "lush_get_inventories") then
			error("you need the 'lush_get_inventories' to access inventories other than yours")
		end

		local inv = minetest.get_inventory(selector_params.location)

		if not inv then
			error("couldnt find inventory")
		end

		if selector_params.wielded then
			if selector_params.location.type ~= "player" then
				error("wielded argument can only work with player inventories")
			end

			output = InventoryItemList:new(
			{
				location = selector_params.location,
				indexes = {[minetest.get_player_by_name(selector_params.location.name):get_wield_list()] = {minetest.get_player_by_name(selector_params.location.name):get_wield_index()}}
			})
		else
			local indexes = {}
			local valid
			for listname, list in pairs(inv:get_lists()) do
				indexes[listname] = {}
				if selector_params.list == listname or not selector_params.list then
					for i, stack in pairs(list) do
						valid = true

						-- could be optimized, see comment about matchers above
						for selector_param_key, selector_param_value in pairs(selector_params) do
							if Selector.Matchers["item"][selector_param_key] and Selector.Matchers["item"][selector_param_key](stack, selector_param_value, context) == (inverted_params[selector_param_key] ~= nil) then
								valid = false
							end
						end

						if valid then
							table.insert(indexes[listname], i)
						end
					end
				end
			end

			output = InventoryItemList:new(
			{
				location = selector_params.location,
				indexes = indexes
			})

		end

		output = {output}
	else
		error("invalid selector_type")
	end

	return output
end

return Selector
