-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local helpers = require("helpers")
local Expression = require("language_constructs.expression")
local StringExpression = require("language_constructs.string_expression")
local Shell = require("language_constructs.shell")
local DatatypeValidator = require("datatype_validator")

local ForLoop =
{
	instance_of = "LanguageConstruct_ForLoop",
}

ForLoop.__index = ForLoop

function ForLoop:new(t)
	t = t or {}
	self = setmetatable(
	{
		param_name = t.param_name or "",
		iterable = t.iterable or "",
		subshell = t.subshell or "",
	}, ForLoop)

	return self
end

function ForLoop:evaluate(context, max_frustration)
	-- {{{ type checking
	if context.instance_of ~= "ShellContext" then error("context should be an instance of ShellContext. instead of " .. context.instance_of) end
	-- }}}

	max_frustration = max_frustration or 10000
	local frustration = 0
	local evaluated_param_name = self.param_name:evaluate(context)
	local evaluated_iterable = self.iterable:evaluate(context)

	if type(evaluated_iterable) == "table" and evaluated_iterable.instance_of == "TypedList" then
		for i, v in evaluated_iterable:iterator() do
			if frustration > max_frustration then
				error("loop exceeded maximum iterations, as defined by lush_max_loop_iterations setting")
			end

			context.env:set(evaluated_param_name, v, false, true)
			self.subshell:evaluate(context, true)

			if context.exit_status ~= 0 then
				error("for loop: " .. context.stderr)
			end

			frustration = frustration + 1
		end
	else
		local iterable_type = DatatypeValidator.get_type_of(evaluated_iterable)

		if not (DatatypeValidator.registered_types[iterable_type] 
			and DatatypeValidator.registered_types[iterable_type].iterator) then
			error("cant be iterated over")
		end

		for v in DatatypeValidator.registered_types[iterable_type].iterator(evaluated_iterable) do
			if frustration > max_frustration then
				error("loop exceeded maximum iterations, as defined by lush_max_loop_iterations setting")
			end

			context.env:set(evaluated_param_name, v, false, true)
			self.subshell:evaluate(context, true)

			if context.exit_status ~= 0 then
				error("for loop: " .. context.stderr)
			end

			frustration = frustration + 1
		end
	end
end

function ForLoop:parse(parser_context)
	-- {{{ type checking
	if parser_context.instance_of ~= "ParserContext" then error("parser_context should be an instance of ParserContext. instead of " .. parser_context.instance_of) end
	-- }}}

	local char

	-- state: *

	local success = parser_context:match_fast("for")

	if not success then
		error("expected 'for'")
	end

	-- state: for*
	
	parser_context:skip_until(helpers.white_chars_set, true)

	self.param_name = StringExpression:new()
	self.param_name:parse(parser_context, helpers.white_chars_set)

	-- state: for <param_name>*

	parser_context:skip_until(helpers.white_chars_set, true)

	local success = parser_context:match_fast("in")

	if not success then
		error("expected 'in'")
	end

	-- state: for <param_name> in*

	parser_context:skip_until(helpers.white_chars_set, true)

	self.iterable = Expression:new()
	self.iterable:parse(parser_context, helpers.white_chars_set)

	-- state: for <param_name> in <iterable>*

	parser_context:skip_until(helpers.white_chars_set, true)

	local success = parser_context:match_fast("do")

	if not success then
		error("expected 'do'")
	end

	-- state: for <param_name> in <iterable> do*

	parser_context:skip_until(helpers.white_chars_set, true)

	self.subshell = Shell:new()
	self.subshell:parse(parser_context, {"end"})

	parser_context:advance(3)
	parser_context:skip_until(helpers.white_chars_set, true)

	-- state: for <param_name> in <iterable> do
	-- <subshell>
	-- end*
end

return ForLoop
