-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local helpers = require("helpers")
local Expression
local StringExpression

local ParameterAsignment =
{
	instance_of = "LanguageConstruct_ParameterAsignment"
}

ParameterAsignment.__index = ParameterAsignment

function ParameterAsignment:new(t)
	if t == nil then
		t = {}
	end

	-- {{{ type checking
	if type(t) ~= "table" then error("t should be of type table. instead of " .. type(t)) end
	-- }}}

	self = setmetatable(
	{
		--- @string name
		--	the name of the parameter
		name = t.name or "",

		--- @param value
		--	the value that the parameter will be set to
		value = t.value or ""
	}, ParameterAsignment)

	return self
end

function ParameterAsignment:evaluate(context)
	-- {{{ type checking
	if context.instance_of ~= "ShellContext" then error("context should be an instance of ShellContext. instead of " .. context.instance_of) end
	-- }}}

	context.env:set(self.name:evaluate(context), self.value:evaluate(context))
end

function ParameterAsignment:parse(parser_context, terminator_set)
	-- {{{ type checking
	if parser_context.instance_of ~= "ParserContext" then error("parser_context should be an instance of ParserContext. instead of " .. parser_context.instance_of) end
	-- }}}

	Expression = Expression or require("language_constructs.expression")
	StringExpression = StringExpression or require("language_constructs.string_expression")

	-- state: *

	self.name = StringExpression:new()
	self.name:parse(parser_context, {["="] = true})

	-- state: <parameter name>*
	if parser_context:consume() ~= "=" then
		error("parameter asignment must have a format of <param_name>=<value>")
	end
	-- state: <parameter name>=*

	self.value = Expression:new()
	self.value:parse(parser_context, terminator_set)

	-- state: <parameter name>=<value>*
end

return ParameterAsignment
