-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local DatatypeValidator = require("datatype_validator")

DatatypeValidator.register_type("string", 
{
	get_type = function(value)
		return type(value) == "string"
	end,

	iterator = function(value)
		i = 0
		return function()
			if i < #value  then 
				i = i + 1
				return value:sub(i, i)
			end
		end
	end,

	to_bool = function(value)
		return #value > 0
	end,

	conversions = 
	{
		["number"] = function(value)
			return tonumber(value)
		end,

		["time"] = function(value)
			return helpers.parse_time(value)
		end,
	},
})

DatatypeValidator.register_type("number", 
{
	get_type = function(value)
		return type(value) == "number"
	end,

	to_bool = function(value)
		return value ~= 0
	end,

	iterator = function(value)
		local i = 0
		return function()
			if i < value then
				i = i + 1
				return i
			end
		end
	end,

	conversions = 
	{
		["string"] = function(value)
			return tostring(value)
		end,

		["time"] = function(value)
			return value
		end,
	},
})

DatatypeValidator.register_type("time", 
{
	get_type = function(value)
		return type(value) == "number"
	end,

	to_bool = function(value)
		return value ~= 0
	end,

	conversions = 
	{
		["string"] = function(value)
			return helpers.time_to_string(value)
		end,

		["number"] = function(value)
			return value
		end,
	},
})

DatatypeValidator.register_type("range",
{
	get_type = function(value)
		return type(value) == "table" and value.instance_of == "Range"
	end,

	iterator = function(value)
		return value:iterator()
	end,

	to_bool = function(value)
		return true
	end,

	conversions = 
	{
		["string"] = function(value)
			return tostring(value)
		end,
	},
})

DatatypeValidator.register_type("table",
{
	get_type = function(value)
		return nil -- detecting tables is a special case thats hardcoded elsewhere
	end,

	to_bool = function(value)
		return #value > 0
	end,

	conversions = 
	{
		["string"] = function(value)
			return dump(value)
		end,
	},
})

