-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests/busted_definitions")

describe("LanguageConstruct_CommandInvocation class: ",
function()
	local CommandInvocation = require("language_constructs.command_invocation")
	local Command = require("command")
	local ShellContext = require("shell_context")
	local ParserContext = require("parser_context")
	local helpers = require("helpers")
	local command_invocation_instance
	local shell_context_instance
	local parser_context_instance
	local example_command
	local command_spy
	local command_terminators =
	{
		[" "] = true,
		["\t"] = true,
		[";"] = true,
		["|"] = true,
	}


	before_each(
	function()
		example_command = Command:new(
		{
			name = "command",
			options = {["h"] = true},
			callback = function(context, args, options)
			end
		})

		command_spy = spy.on(example_command, "callback")

		command_invocation_instance = CommandInvocation:new()

		shell_context_instance = ShellContext:new()
		parser_context_instance = ParserContext:new()

		shell_context_instance.env["parameter"] = "value"
	end)

	after_each(
	function()
		example_command:unregister()
	end)

	describe("evaluates correctly: ",
	function()

		it("when given simple args",
		function()
			command_invocation_instance.name = helpers.create_string_expression("command")
			command_invocation_instance.args =
			{
				helpers.create_string_expression("arg1"),
				helpers.create_string_expression("arg2"),
				helpers.create_string_expression("arg3")
			}

			command_invocation_instance:evaluate(shell_context_instance)

			assert.spy(command_spy).was_called_with(match.is_ref(shell_context_instance), match.same({"arg1", "arg2", "arg3"}), match.same({}))
		end)

		it("when given args with options",
		function()
			command_invocation_instance.name = helpers.create_string_expression("command")
			command_invocation_instance.args =
			{
				helpers.create_string_expression("arg1"),
				helpers.create_string_expression("-h"),
				helpers.create_string_expression("option_arg"),
				helpers.create_string_expression("arg2")
			}

			command_invocation_instance:evaluate(shell_context_instance)

			assert.spy(command_spy).was_called_with(match.is_ref(shell_context_instance), match.same({"arg1", "arg2"}), match.same({["h"] = "option_arg"}))
		end)

		it("when given a parameter expansion",
		function()
			command_invocation_instance.name = helpers.create_string_expression("command")
			command_invocation_instance.args =
			{
				helpers.create_string_expression("$parameter"),
			}

			command_invocation_instance:evaluate(shell_context_instance)

			assert.spy(command_spy).was_called_with(match.is_ref(shell_context_instance), match.same({"value"}), match.same({}))
		end)

		it("errors when trying to call a non existent command",
		function()
			command_invocation_instance.name = helpers.create_string_expression("nonexistent_command")
			command_invocation_instance.args = {}

			assert.errors(function() command_invocation_instance:evaluat(shell_context_instance) end)
		end)
	end)

	describe("parses correctly: ",
	function()

		it("when given unquoted strings as args",
		function()
			parser_context_instance.text = [[command arg1 arg2 $parameter]]

			command_invocation_instance:parse(parser_context_instance, command_terminators)

			assert.equals("command", command_invocation_instance.name:evaluate(shell_context_instance))

			assert.equals("arg1", command_invocation_instance.args[1]:evaluate(shell_context_instance))
			assert.equals("arg2", command_invocation_instance.args[2]:evaluate(shell_context_instance))
			assert.equals("value", command_invocation_instance.args[3]:evaluate(shell_context_instance))

			assert.equals(29, parser_context_instance.character_index)
		end)

		it("when given double quoted strings as args",
		function()
			parser_context_instance.text = [[command "arg1" "arg2" "parameter: $parameter"]]

			command_invocation_instance:parse(parser_context_instance, command_terminators)

			assert.equals("command", command_invocation_instance.name:evaluate(shell_context_instance))

			assert.equals("arg1", command_invocation_instance.args[1]:evaluate(shell_context_instance))
			assert.equals("arg2", command_invocation_instance.args[2]:evaluate(shell_context_instance))
			assert.equals("parameter: value", command_invocation_instance.args[3]:evaluate(shell_context_instance))

			assert.equals(46, parser_context_instance.character_index)
		end)

		it("when given single quoted strings as args",
		function()
			parser_context_instance.text = [[command 'arg1' 'arg2' '$parameter']]

			command_invocation_instance:parse(parser_context_instance, command_terminators)

			assert.equals("command", command_invocation_instance.name:evaluate(shell_context_instance))

			assert.equals("arg1", command_invocation_instance.args[1]:evaluate(shell_context_instance))
			assert.equals("arg2", command_invocation_instance.args[2]:evaluate(shell_context_instance))
			assert.equals("$parameter", command_invocation_instance.args[3]:evaluate(shell_context_instance))

			assert.equals(35, parser_context_instance.character_index)
		end)

		it("when the name itself is a parameter expansion",
		function()
			parser_context_instance.text = [[$parameter arg]]

			command_invocation_instance:parse(parser_context_instance, command_terminators)

			assert.equals("value", command_invocation_instance.name:evaluate(shell_context_instance))

			assert.equals("arg", command_invocation_instance.args[1]:evaluate(shell_context_instance))

			assert.equals(15, parser_context_instance.character_index)
		end)

		it("ends parsing when encountering characters in the CommandInvocation_Terminators table",
		function()
			parser_context_instance.text = [[command arg || command2]]

			command_invocation_instance:parse(parser_context_instance, command_terminators)

			assert.equals("command", command_invocation_instance.name:evaluate(shell_context_instance))

			assert.equals("arg", command_invocation_instance.args[1]:evaluate(shell_context_instance))

			assert.equals(13, parser_context_instance.character_index)
		end)

		it("ends parsing when encountering a new line",
		function()
			parser_context_instance.text = "command arg \n command2"

			command_invocation_instance:parse(parser_context_instance, command_terminators)

			assert.equals("command", command_invocation_instance.name:evaluate(shell_context_instance))

			assert.equals("arg", command_invocation_instance.args[1]:evaluate(shell_context_instance))

			assert.equals(13, parser_context_instance.character_index)
		end)
	end)

	describe("all members are correctly set by new(): ",
	function()
		local definition_table

		before_each(
		function()
			definition_table =
			{
				name = helpers.create_string_expression("command"),
				args =
				{
					helpers.create_string_expression("arg1"),
					helpers.create_string_expression("arg2"),
					helpers.create_string_expression("arg3"),
				}
			}
		end)

		it("name",
		function()
			command_invocation_instance = CommandInvocation:new(definition_table)

			assert.equals(definition_table.name, command_invocation_instance.name)
		end)

		it("args",
		function()
			command_invocation_instance = CommandInvocation:new(definition_table)

			assert.equals(definition_table.args, command_invocation_instance.args)
		end)
	end)
end)
