-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_RangeExpression
--	describes a vector

local inspect = require("inspect")
local helpers = require("helpers")
local Range = require("range")
local NumericExpression

local RangeExpression =
{
	instance_of = "LanguageConstruct_RangeExpression",
}

RangeExpression.__index = RangeExpression

function RangeExpression:new(t)
	t = t or {}
	self = setmetatable(
	{
		start = t.members or -math.huge,
		finish = t.members or math.huge,
		start_inclusive = t.start_inclusive or true,
		finish_inclusive = t.finish_inclusive or true,
	}, RangeExpression)

	return self
end

function RangeExpression:evaluate(context)
	-- {{{ type checking
	if context.instance_of ~= "ShellContext" then error("context should be an instance of ShellContext. instead of " .. context.instance_of) end
	-- }}}
	local output = Range:new(
	{
		start_inclusive = self.start_inclusive,
		finish_inclusive = self.finish_inclusive
	})

	if self.start == -math.huge then
		output.start = -math.huge
	else
		output.start = self.start:evaluate(context)
	end

	if self.finish == math.huge then
		output.finish = math.huge
	else
		output.finish = self.finish:evaluate(context)
	end

	return output
end

-- the state comments dont include white spaces because it would be too verbose to be useful
function RangeExpression:parse(parser_context)
	-- {{{ type checking
	if parser_context.instance_of ~= "ParserContext" then error("parser_context should be an instance of ParserContext. instead of " .. parser_context.instance_of) end
	-- }}}

	NumericExpression = NumericExpression or require("language_constructs.numeric_expression")
	self.start = -math.huge
	self.finish = math.huge

	local char = parser_context:consume()

	if char == "(" then
		self.start_inclusive = false
	elseif char == "<" then
		self.start_inclusive = true
	else
		error([[range expression expected to start with a "(" or "<", instead got ]] .. char)
	end

	-- state: <opening char>*

	parser_context:skip_until(helpers.white_chars_set, true)
	char = parser_context:peek()

	if char ~= ":" then
		self.start = NumericExpression:new()
		self.start:parse(parser_context, {[":"] = true})
	end

	-- state: <opening char><numeric expression>*

	parser_context:skip_until(helpers.white_chars_set, true)
	char = parser_context:consume()

	if char ~= ":" then
		error([[range boandaries are supposed to be seperated by ":", instead of ]] .. char)
	end

	-- state: <opening char><numeric expression>:*

	parser_context:skip_until(helpers.white_chars_set, true)
	char = parser_context:peek()

	if char ~= ")" and char ~= ">" then
		self.finish = NumericExpression:new()
		self.finish:parse(parser_context, {[")"] = true, [">"] = true})
	end

	-- state: <opening char><numeric expression>:*<numeric expression>

	parser_context:skip_until(helpers.white_chars_set, true)
	char = parser_context:consume()

	if char == ")" then
		self.finish_inclusive = false
	elseif char == ">" then
		self.finish_inclusive = true
	else
		error([[range expression expected to finish with a ")" or ">", instead got ]] .. char)
	end
end

return RangeExpression
