-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_VectorExpression
--	describes a vector

local inspect = require("inspect")
local helpers = require("helpers")
local NumericExpression
local NumberLiteral
local member_terminator_set =
{
	[","] = true,
	[" "] = true
}

local VectorExpression =
{
	instance_of = "LanguageConstruct_VectorExpression",
}

VectorExpression.__index = VectorExpression

function VectorExpression:new(t)
	t = t or {}
	self = setmetatable(
	{
		members = t.members or {0, 0, 0},
		relative = t.members or {false, false, false},
		caret_notation = t.caret_notation or false
	}, VectorExpression)

	return self
end

function VectorExpression:evaluate(context)
	-- {{{ type checking
	if context.instance_of ~= "ShellContext" then error("context should be an instance of ShellContext. instead of " .. context.instance_of) end
	-- }}}
	local output = {}
	for idx = 1, 3 do
		if self.caret_notation then
			-- @todo implement caret notation
		end

		-- if is relative to player's position
		if self.relative[idx] then
			table.insert(output, self.members[idx]:evaluate(context) + context.env:get("pos")[idx])
		else
			table.insert(output, self.members[idx]:evaluate(context))
		end
	end

	return output
end

function VectorExpression:parse(parser_context)
	-- {{{ type checking
	if parser_context.instance_of ~= "ParserContext" then error("parser_context should be an instance of ParserContext. instead of " .. parser_context.instance_of) end
	-- }}}

	NumericExpression = NumericExpression or require("language_constructs.numeric_expression")
	NumberLiteral = NumberLiteral or require("language_constructs.number_literal")

	local char = parser_context:consume()
	local terminator_set = member_terminator_set

	if char == "^" then
		self.caret_notation = true
		if parser_context:consume() ~= "("  then
			error("expected \"(\", instead of " .. char)
		end
	elseif char ~= "(" then
		error("expected \"(\", instead of " .. char)
	end

	for idx = 1, 3 do
		-- state: *
		char = parser_context:peek()

		if char == "~" then
			if self.caret_notation then
				error("caret notation cant be combined with relative vectors")
			end

			self.relative[idx] = true
			parser_context:advance()
			-- state: ~*
		end

		-- state: <tilde?>*


		if idx == 3 then
			-- for the third iteration, add ) to the terminator set
			terminator_set = helpers.concat_sets(member_terminator_set, {[")"] = true})
		end

		char = parser_context:peek()

		if terminator_set[char] then
			-- the number expression is missing, defaulting to 0
			self.members[idx] = NumberLiteral:new({value = 0})
		else
			self.members[idx] = NumericExpression:new()
			self.members[idx]:parse(parser_context, terminator_set)
		end

		-- state: <tilde?><numeric expression?>*

		if self.members[idx] == nil then
			error("invalid vector memeber")
		end

		parser_context:skip_until(member_terminator_set, true)

		-- state: <tilde?><numeric expression><peroids and spaces>*
	end

	char = parser_context:consume()

	if char ~= ")" then
		error("expected \")\" to close vector, instead got " .. char)
	end
end

return VectorExpression
