-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local DatatypeValidator = require("datatype_validator")

require("type_definitions")
local helpers = require("helpers")

DatatypeValidator.register_type("node_name", 
{
	get_type = function(value)
		return type(value) == "string" and helpers.normalize_nodename(value) ~= nil
	end,

	to_bool = function(value)
		return #value > 0
	end,

	conversions = 
	{
		["string"] = function(value)
			return value
		end,
	},
})

DatatypeValidator.registered_types["string"].conversions["node_name"] = function(value)
	return helpers.normalize_nodename(value)
end

DatatypeValidator.register_type("inventory_item_list",
{
	get_type = function(value)
		return type(value) == "table" and value.instance_of == "InventoryItemList"
	end,

	to_bool = function(value)
		return true
	end,

	conversions = 
	{
		["string"] = function(value)
			return tostring(value)
		end,
	},
})

DatatypeValidator.register_type("vector",
{
	get_type = function(value)
		return vector.check(value)
	end,

	to_bool = function(value)
		return true
	end,

	conversions = 
	{
		["string"] = function(value)
			return tostring(value)
		end
	}
})

DatatypeValidator.registered_types["table"].conversions["vector"] = function(value, context)
	if #value == 3 then
		for i = 1, 3 do
			if type(value[i]) ~= "number" then
				return false
			end

			return vector.new(value[1], value[2], value[3])
		end
	end
end

DatatypeValidator.registered_types["string"].conversions["vector"] = function(value)
		converted_value = vector.from_string(value)
		if converted_value then
			return converted_value, true
		elseif minetest.get_player_by_name(value) then
			return minetest.get_player_by_name(value):get_pos(), true
		end
end

DatatypeValidator.register_type("objectref", 
{
	get_type = function(value)
		return value.get_pos and value:get_pos() ~= nil
	end,

	to_bool = function(value)
		return value.get_pos and value:get_pos() ~= nil
	end,

	conversions = 
	{
		["string"] = function(value)
			if value:is_player() then
				return value:get_player_name()
			elseif value.nametag then
				return value.nametag
			else
				return tostring(value)
			end
		end,

		["vector"] = function(value, context)
			-- if tries to convert to position (other than himself) but dosent have the required privilege
			if not context or (value:is_player() and context.env:get("name") ~= value:get_player_name())
				and not minetest.check_player_privs(context.env:get("name"), "lush_get_object_pos") then
				minetest.log("tried to convert ObjectRef to position, refused to do so")
				return nil
			end

			return value:get_pos()
		end,

		["player_name"] = function(value)
			if value:is_player() then
				return value:get_player_name()
			end
		end,

		["player"] = function(value)
			if value:is_player() then
				return value
			end
		end,
	},
})

DatatypeValidator.registered_types["string"].conversions["objectref"] = function(value)
	return minetest.get_player_by_name(value)
end,

DatatypeValidator.register_type("player_name",
{
	get_type = function(value)
		return type(value) == "string" and minetest.get_player_by_name(value) ~= nil
	end,

	to_bool = function(value)
		return #value > 0
	end,

	conversions = 
	{
		["string"] = function(value)
			return value
		end,

		["objectref"] = function(value)
			return minetest.get_player_by_name(value)
		end,
	},
})

DatatypeValidator.registered_types["string"].conversions["player_name"] = function(value)
	if minetest.get_player_by_name(value) then
		return value
	end
end

DatatypeValidator.register_type("player",
{
	get_type = function(value)
		return value.get_pos and value:get_pos() and value:is_player()
	end,

	to_bool = function(value)
		return value.get_pos and value:get_pos() and value:is_player()
	end,

	conversions = 
	{
		["string"] = function(value)
			return value:get_player_name()
		end,

		["player_name"] = function(value)
			return value:get_player_name()
		end,

		["objectref"] = function(value)
			return value
		end,

		["vector"] = function(value, context)
			-- if tries to convert to position (other than himself) but dosent have the required privilege
			if not context or (context.env:get("name") ~= value:get_player_name())
				and not minetest.check_player_privs(context.env:get("name"), "lush_get_object_pos") then
				minetest.log("tried to convert player reference to a position, refused to do so")
				return nil
			end

			return value:get_pos()
		end,
	},
})

DatatypeValidator.registered_types["string"].conversions["player"] = function(value)
	return minetest.get_player_by_name(value)
end
