-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local helpers = require("helpers")
local CommandInvocation
local ShellContext
local InlineLua
local ParameterAsignment
local TypedList
local ForLoop
local WhileLoop
local IfStatement
local argument_terminators =
{
	[" "] = true,
	["\t"] = true,
	[";"] = true,
	["|"] = true,
}


local control_operators =
{
	sequential = 0,
	pipe = 1,
	success = 2,
	failure = 3
}

local Shell =
{
	instance_of = "LanguageConstruct_Shell"
}

Shell.__index = Shell

function Shell:new(t)
	if t == nil then
		t = {}
	end

	-- {{{ type checking
	if type(t) ~= "table" then error("t should be of type table. instead of " .. type(t)) end
	-- }}}

	self = setmetatable(
	{
		--- list where the first elements is a language construct (like a command invocation)
		--	and the elements afterwards is an enum type that signifies the operator (pipe, or, and, ; )
		--	this is kinda a weird, but its the most straightforward way
		actions = t.actions or {}
	}, Shell)

	return self
end

function Shell:evaluate(context, share_environment)
	-- {{{ type checking
	if context.instance_of ~= "ShellContext" then error("context should be an instance of ShellContext. instead of " .. context.instance_of) end
	-- }}}

	TypedList = TypedList or require("typed_list")
	ShellContext = ShellContext or require("shell_context")


	local subshell_context = ShellContext:new()
	subshell_context.stdin:concat(context.stdin)

	if share_environment then
		subshell_context.env = context.env
	else
		subshell_context.env:inherit(context.env)
	end

	local current_index = 1
	local operator = control_operators.sequential
	local success
	local error_message
	while current_index < #self.actions do
		if operator == control_operators.sequential or (subshell_context.exit_status ~= 0 and operator == control_operators.failure) or (subshell_context.exit_status == 0 and operator ~= control_operators.failure) then
			if self.actions[current_index].instance_of == "LanguageConstruct_Shell" then
				-- special behaviour so subshells share the environment
				success, error_message = pcall(self.actions[current_index].evaluate, self.actions[current_index], subshell_context, true)
			else
				success, error_message = pcall(self.actions[current_index].evaluate, self.actions[current_index], subshell_context)
			end

			current_index = current_index + 1
			operator = self.actions[current_index]

			if success then
				subshell_context.exit_status = 0
				subshell_context.stderr = ""
			else
				subshell_context.exit_status = 1
				subshell_context.stderr = error_message
				context.stdout:push(error_message)
			end

			if success and operator == control_operators.pipe then
				subshell_context.stdin = subshell_context.stdout
				subshell_context.stdout = TypedList:new()
			else
				context.stdout:concat(subshell_context.stdout)
				subshell_context.stdin:clear()
				subshell_context.stdout:clear()
			end


		end

		current_index = current_index + 1
	end

	context.stderr = subshell_context.stderr
	context.exit_status = subshell_context.exit_status
end

-- in this case we are using a terminator_list and a terminator_set
-- this is a hacky solution to the problem that i want to terminate a shell when an argument ends with )
-- but also specify terminators with more characters like "end", so all these can be implemented as a subshell
-- (command arg)
-- (command arg )
-- if x then command arg end
function Shell:parse(parser_context, terminator_list, terminator_set)
	-- {{{ type checking
	if parser_context.instance_of ~= "ParserContext" then error("parser_context should be an instance of ParserContext. instead of " .. parser_context.instance_of) end
	-- }}}

	CommandInvocation = CommandInvocation or require("language_constructs.command_invocation")
	InlineLua = InlineLua or require("language_constructs.inline_lua")
	ParameterAsignment = ParameterAsignment or require("language_constructs.parameter_asignment")
	ForLoop = ForLoop or require("language_constructs.for_loop")
	IfStatement = IfStatement or require("language_constructs.if_statement")
	WhileLoop = WhileLoop or require("language_constructs.while_loop")

	terminator_set = helpers.concat_sets({[";"] = true, ["|"] = true}, terminator_set)

	local action
	local matched
	local operator
	local char
	while not parser_context:is_EOF() do
		parser_context:skip_until(helpers.white_chars_set, true)

		char = parser_context:peek()

		if char == "(" then -- expected: subshell
			parser_context:advance()
			action = Shell:new()
			action:parse(parser_context, {")"}, helpers.concat_sets({[")"] = true}, argument_terminators))
		elseif char == "`" then -- expected: inline lua code
			action = InlineLua:new()
			action:parse(parser_context)
		elseif string.find(parser_context.text, "^for", parser_context.character_index) then -- expected: for loop
			action = ForLoop:new()
			action:parse(parser_context)
		elseif string.find(parser_context.text, "^if", parser_context.character_index) then -- expected: if statement
			action = IfStatement:new()
			action:parse(parser_context)
		elseif string.find(parser_context.text, "^while", parser_context.character_index) then -- expected: if statement
			action = WhileLoop:new()
			action:parse(parser_context)
		elseif string.find(parser_context.text, "^[a-zA-Z_][a-zA-Z0-9_]*=", parser_context.character_index) then -- expected: parameter asignment
			action = ParameterAsignment:new()
			action:parse(parser_context, terminator_set)
		else
			action = CommandInvocation:new()
			action:parse(parser_context, terminator_set)
		end

		table.insert(self.actions, action)

		-- state: <previous action if any><action terminator><whitespace><action>*

		parser_context:skip_until(helpers.white_chars_set, true)
		matched, operator = parser_context:match(helpers.control_operators_list, true)

		-- if not a control operator
		if not matched and not parser_context:is_EOF() then
			-- print(parser_context.character_index)
			-- print(parser_context:peek())
			-- print(inspect(terminator_set))
			if terminator_set[parser_context:consume()] then
				-- state: <previous iteration><whitespaces><action><shell terminator>*
				table.insert(self.actions, control_operators.sequential)
				return
			end

			-- if the parser got to this point
			-- this means that a the action that was just parsed ended parsing in an unexpected location
			-- this means everything broke
			error("OH SHIT OH FUCK!!!!!\n" .. action.instance_of .. " fucked up BIG time")
		end

		-- state: <previous iteration><whitespaces><action><control operator>*

		if operator == "|" then
			table.insert(self.actions, control_operators.pipe)
		elseif operator == "&&" then
			table.insert(self.actions, control_operators.success)
		elseif operator == "||" then
			table.insert(self.actions, control_operators.failure)
		else
			table.insert(self.actions, control_operators.sequential)
		end

		-- checking if the shell should be terminated by a keyword
		parser_context:skip_until(helpers.white_chars_set, true)
		local matched, match = parser_context:match(terminator_list)

		-- state: <previous iteration><whitespaces><action><control operator><whitespaces><terminator>*
		if matched then
			break
		end
	end

	--	handling an edgecase where a shell ends with a control operator, like
	--	echo text |
	--	but allow something like this
	--	echo text;
	if self.actions[#self.actions] ~= control_operators.sequential then
		error("unexpected control operator")
	end
end

return Shell
