-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- this file contains simple commands that dont depend on minetest namespace

minetest.register_privilege("export",
{
	description = "whether the player can modify lush's global environment",
	give_to_singleplayer = true
})

lush.Command:new(
{
	name = "echo",
	description = "prints arguments to STDOUT",
	params = "<arguments>",
	stdin_type = "string",
	callback = function(context, args, options)
		for i, v in pairs(args) do
			if type(v) == "table" and v.instance_of == "TypedList" then
				for _, v2 in v:iterator() do
					context.stdout:push(v2)
				end
			else
				context.stdout:push(v)
			end
		end
	end,
})

lush.Command:new(
{
	name = "type",
	description = "prints the type of the argument specified",
	arguments = 
	{
		{
			name = "value",
			single = true
		}
	},
	callback = function(context, args, options)
		context.stdout:push(lush.DatatypeValidator.get_type_of(args.value))
	end,
})

lush.Command:new(
{
	name = "cast",
	description = "casts a value to the specified type",
	arguments = 
	{
		{
			name = "value",
		},
		{
			name = "type",
			type = "string",
			single = true
		}
	},
	callback = function(context, args, options)
		local success = args.value:cast_to(args.type, context)

		if not success then
			error(string.format("cant convert from '%s' to '%s'", args.value.type, args.type))
		end

		context.stdout = args.value
	end,
})

lush.Command:new(
{
	name = "apply_ruleset",
	description = "interprets the list and converts it a format specified by the ruleset. into meant for argument validation",
	privs = {debug = true},
	arguments = 
	{
		{
			name = "list",
			type = "table",
			single = true
		},
		{
			name = "ruleset",
			type = "table",
			single = true
		}
	},
	callback = function(context, args, options)
		local output = lush.DatatypeValidator.validate(args.list, args.ruleset, context)

		context.stdout:push(output)
	end,
})

lush.Command:new(
{
	name = "eval_command_args",
	description = "returns the args that are passed to a command for debug purposes, dosent work well with options",
	arguments = 
	{
		{
			name = "command_name",
			type = "string",
			single = true
		},
		{
			rule_type = "block",
			name = "arg",
		}
	},
	callback = function(context, args, options)
		minetest.debug(dump(args))
		local output = lush.DatatypeValidator.validate(args.arg, lush.RegisteredCommands[args.command_name].arguments, context)

		context.stdout:push(output)
	end,
})

lush.Command:new(
{
	name = "append",
	description = "appends a string to every line in STDIN",
	arguments =
	{
		{
			name = "postfix",
			type = "string",
			single = true
		}
	},
	stdin_type = "string",
	callback = function(context, args, options)
		for i, v in pairs(context.stdin.container) do
			context.stdout:push(v .. args.postfix)
		end
	end,
})

lush.Command:new(
{
	name = "lines",
	description = "takes STDIN and prints only lines in the range",
	arguments =
	{
		{
			name = "range",
			type = "range",
			single = true
		}
	},
	callback = function(context, args, options)
		for line_num, line in context.stdin:iterator() do
			if args.range:contains(line_num) then
				context.stdout:push(line)
			end
		end
	end,
})

lush.Command:new(
{
	name = "wc",
	description = "counts the amount of lines in STDIN",
	arguments = {},
	callback = function(context, args, options)
		context.stdout:push(#context.stdin.container)
	end,
})

lush.Command:new(
{
	name = "prepend",
	description = "prepends all lines from STDIN with the given string",
	arguments =
	{
		{
			name = "prefix",
			type = "string",
			single = true
		}
	},
	stdin_type = "string",
	callback = function(context, args, options)
		for i, v in pairs(context.stdin.container) do
			context.stdout:push(args.prefix .. v)
		end
	end,
})

lush.Command:new(
{
	name = "set",
	description = "sets the given parameter to the value of STDIN",
	arguments =
	{
		{
			name = "param",
			type = "string",
			single = true
		}
	},
	callback = function(context, args, options)
		context.env[args.param] = context.stdin:get(1)
	end,
})

lush.Command:new(
{
	name = "split",
	description = "splits STDIN into lines",
	arguments =
	{
		{
			name = "seperator",
			type = "string",
			single = true
		}
	},
	stdin_type = "string",
	callback = function(context, args, options)
		for i, v in pairs(context.stdin.container) do
			context.stdout:concat(lush.helpers.string_split(v, args.seperator))
		end
	end,
})

lush.Command:new(
{
	name = "match",
	description = "prints only the lines from STDIN that match the pattern",
	arguments =
	{
		{
			name = "pattern",
			type = "string",
			single = true
		}
	},
	stdin_type = "string",
	callback = function(context, args, options)
		for i, v in pairs(context.stdin.container) do
			if string.find(v, args.pattern) then
				context.stdout:push(v)
			end

		end
	end,
})

lush.Command:new(
{
	name = "exit",
	description = "exits with the specified exit code",
	arguments =
	{
		{
			name = "exit_status",
			type = "number",
			single = true
		}
	},
	callback = function(context, args, options)
		if args.exit_status ~= 0 then
			error()
		end
	end,
})

lush.Command:new(
{
	name = "vecdesc",
	description = "verbosely prints information about a vector",
	arguments =
	{
		{
			name = "vector",
			type = "vector",
			single = true
		}
	},
	callback = function(context, args, options)
		context.stdout:push("x: " .. tostring(args.vector.x))
		context.stdout:push("y: " .. tostring(args.vector.y))
		context.stdout:push("z: " .. tostring(args.vector.z))
	end,
})

lush.Command:new(
{
	name = "lush",
	description = "runs a shell",
	arguments =
	{
		{
			name = "code",
			type = "string",
			single = true,
			optional = true
		}
	},
	callback = function(context, args, options)
		lush.Interpreter.run_shell(args.code, lush.ShellContext:new(lush.global_env))
	end,
})

lush.Command:new(
{
	name = "export",
	description = "exports the parameter to the global environment, where every player can access it. if no value is provided, it unsets the global variable instead",
	options =
	{
		["read-only"] = false,
		["volatile"] = false,
	},
	privs = {export = true},
	arguments =
	{
		{
			name = "param",
			type = "string",
			single = true
		}
		-- manually checking the last argument
		-- @todo implement this as an optional argument when rules support it
	},
	callback = function(context, args, options)
		if not args[1] then
			lush.global_env:unset(args.param)
		else
			lush.global_env:set(args.param, args.value, options["read-only"], options["volatile"])
		end
	end,
})

lush.Command:new(
{
	name = "env",
	description = "prints the entire environment to STDOUT",
	callback = function(context, args, options)
		if #args ~= 0 then
			error("invalid arguments: too many arguments")
		end

		local function print_env(context, env)
			if env.inherited_env.instance_of == "Env" then
				print_env(context, env.inherited_env)
			end

			for parameter, value in pairs(env.parameters) do
				if type(value) == "function" then
					context.stdout:push(parameter .. "=" .. lush.DatatypeValidator.cast_to(value(env), "string", context))
				else
					context.stdout:push(parameter .. "=" .. lush.DatatypeValidator.cast_to(value, "string", context))
				end
			end

			for parameter, value in pairs(env.volatile_parameters) do
				if type(value) == "function" then
					context.stdout:push(parameter .. "=" .. lush.DatatypeValidator.cast_to(value(env), "string", context))
				else
					context.stdout:push(parameter .. "=" .. lush.DatatypeValidator.cast_to(value, "string", context))
				end
			end
		end

		print_env(context, context.env)
	end,
})
