-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @submodule Command

local Command, RegisteredCommands = require("command")
local Interpreter = require("interpreter")
local helpers = require("helpers")
local ShellContext = require("shell_context")
local ParserContext = require("parser_context")
local Expression = require("language_constructs.expression")
local DatatypeValidator = require("datatype_validator")

local inspect = require("inspect")

local old_constructor = Command.new
Command.new = function(self, t)
	-- minetest.log(inspect(t))
	-- minetest.log(inspect(Command.is_name_registered(t.name)))
	self = old_constructor(self, t)

	-- {{{ type checking
	if t.privs and type(t.privs) ~= "table" then error("t.privs should be of type table. instead of " .. type(t.privs)) end
	if t.params and type(t.params) ~= "string" then error("t.params should be of type string. instead of " .. type(t.params)) end
	if t.description and type(t.description) ~= "string" then error("t.description should be of type string. instead of " .. type(t.description)) end
	-- }}}

	self.privs = t.privs or {}
	self.params = t.params or ""
	self.description = t.description or {}
	self.mod_origin = minetest.get_current_modname()

	if self.arguments and self.params == "" then
		self.params = DatatypeValidator.generate_param_hint(self.arguments)
	end

	if minetest.registered_chatcommands[self.name] then
		minetest.log("info", string.format("%s overriden by lush", self.name))
		minetest.unregister_chatcommand(self.name)
	end

	self:register_compatibility_command()

	return self
end

local old_unregister = Command.unregister
Command.unregister = function(self)
	old_unregister(self)
	minetest.unregister_chatcommand(self.name)
end

local old_call = Command.call
Command.call = function(self, context, args)
	local has_privs = true
	for k, v in pairs(self.privs) do
		if v ~= context.privilege_cache[k] then
			has_privs = false
			break
		end
	end

	if has_privs then
		old_call(self, context, args)
	else
		local has, missing = minetest.check_player_privs("singleplayer", self.privs)
		error("player is missing privilages: " .. table.concat(missing, ", "))
	end
end

Command.register_compatibility_command = function(self)
	minetest.register_chatcommand(self.name,
	{
		privs = self.privs,
		params = self.params,
		description = self.description,
		func = function(name, params)
			local context = ShellContext:new({env = lush.player_environments[name]})
			local success, message = Interpreter.run_command(self.name .. " " .. params, context)

			if success then
				context.stdout:cast_to("string")
				return true, table.concat(context.stdout.container, "\n")
			else
				return false, message
			end
		end
	})
end

minetest.register_on_chat_message(
function(name, message)
	if string.sub(message, 1, 1) == "!" then
		minetest.log("action", name .. " invoked a lush shell: \n" .. string.sub(message, 2))
		local context = ShellContext:new({env = lush.player_environments[name]})
		Interpreter.run_shell(string.sub(message, 2), context, true,
		function(success, error_msg, context, parse_time, eval_time)
			minetest.chat_send_player(name, minetest.colorize("orange", "invoked shell:"))
			if not success then
				minetest.chat_send_player(name, error_msg)
			else
				minetest.debug(dump(context))
				context.stdout:cast_to("string")
				for _, line in context.stdout:iterator() do
					minetest.chat_send_player(name, line)
				end

				minetest.chat_send_player(name, minetest.colorize("orange", 
					string.format("(%sms, %sms)", parse_time / 1000, eval_time / 1000)))

				minetest.log("action", "exit code: " .. tostring(context.exit_status)
				.. "\nparsing time: " .. tostring(parse_time / 1000) .. "ms"
				.. "\nevaluation time: " .. tostring(eval_time / 1000) .. "ms")
			end
		end)

		return true
	end
end)

minetest.register_on_mods_loaded(function()
	for command, def in pairs(RegisteredCommands) do
		def:register_compatibility_command()
		minetest.registered_chatcommands[command].mod_origin = def.mod_origin
	end
end)

return Command, RegisteredCommands
