-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests/busted_definitions")
local inspect = require("inspect")

describe("LanguageConstruct_VectorExpression class:",
function()
	local VectorExpression = require("language_constructs.vector_expression")
	local ShellContext = require("shell_context")
	local ParserContext = require("parser_context")
	local shell_context_instance
	local parser_context_instance
	local vector_expression_instance

	before_each(
	function()
		shell_context_instance = ShellContext:new()
		parser_context_instance = ParserContext:new()
		vector_expression_instance = VectorExpression:new()
	end)

	describe("parses correctly",
	function()

		it("when given a simple vector",
		function()
			parser_context_instance.text = "(1, 2, 3)"

			vector_expression_instance:parse(parser_context_instance)

			assert.equals(1, vector_expression_instance.members[1]:evaluate(shell_context_instance))
			assert.equals(2, vector_expression_instance.members[2]:evaluate(shell_context_instance))
			assert.equals(3, vector_expression_instance.members[3]:evaluate(shell_context_instance))

			assert.equals(false, vector_expression_instance.caret_notation)
			assert.same({false, false, false}, vector_expression_instance.relative)

			assert.equals(10, parser_context_instance.character_index)
		end)

		it("when given a relative vector",
		function()
			parser_context_instance.text = "(~1, ~2, ~3)"

			vector_expression_instance:parse(parser_context_instance)

			assert.equals(1, vector_expression_instance.members[1]:evaluate(shell_context_instance))
			assert.equals(2, vector_expression_instance.members[2]:evaluate(shell_context_instance))
			assert.equals(3, vector_expression_instance.members[3]:evaluate(shell_context_instance))

			assert.equals(false, vector_expression_instance.caret_notation)
			assert.same({true, true, true}, vector_expression_instance.relative)

			assert.equals(13, parser_context_instance.character_index)
		end)

		it("when given vectors with caret notation",
		function()
			parser_context_instance.text = "^(1, 2, 3)"

			vector_expression_instance:parse(parser_context_instance)

			assert.equals(1, vector_expression_instance.members[1]:evaluate(shell_context_instance))
			assert.equals(2, vector_expression_instance.members[2]:evaluate(shell_context_instance))
			assert.equals(3, vector_expression_instance.members[3]:evaluate(shell_context_instance))

			assert.equals(true, vector_expression_instance.caret_notation)
			assert.same({false, false, false}, vector_expression_instance.relative)

			assert.equals(11, parser_context_instance.character_index)
		end)

		it("errors when combining caret notation with relative vectors",
		function()
			parser_context_instance.text = "^(~1, ~2, ~3)"

			assert.errors(function()
				vector_expression_instance:parse(parser_context_instance)
			end)
		end)

		it("errors when the last character isnt a )",
		function()
			parser_context_instance.text = "^(1, 2, 3%"

			assert.errors(function()
				vector_expression_instance:parse(parser_context_instance)
			end)
		end)

		it("errors when the first isnt a ^ or (",
		function()
			parser_context_instance.text = "%1, 2, 3)"

			assert.errors(function()
				vector_expression_instance:parse(parser_context_instance)
			end)
		end)

		it("errors when the first is a ^ and the second isnt (",
		function()
			parser_context_instance.text = "^%1, 2, 3)"

			assert.errors(function()
				vector_expression_instance:parse(parser_context_instance)
			end)
		end)

		it("errors when members arent valid numbers",
		function()
			parser_context_instance.text = "(a, b, c)"

			assert.errors(function()
				vector_expression_instance:parse(parser_context_instance)
			end)
		end)
	end)
end)
