-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

-- This mod is currently WIP because functions are not implemented yet in lush

local gui = flow.widgets
local storage = minetest.get_mod_storage()

minetest.register_privilege("lush_display",
{
	description = "can display GUI elements on other player's screens",
})


local function create_pager_scroll_area(stdin)
	local output = {}

	for i, v in stdin:iterator() do
		table.insert(output, gui.Label{label = lush.DatatypeValidator.cast_to(v, "string")})
	end

	return output
end

lush.Command:new(
{
	name = "pager",
	description = "displays STDIN in a gui pager",
	options = {prompt = true},
	privs = {},
	arguments = {},
	callback = function(context, args, options)
		local player
		if options.player and context.priv_cache["lush_display"] then
			player = lush.DatatypeValidator.cast_to(options["player"], "player")
		else
			player = minetest.get_player_by_name(context.env:get("name"))
		end

		context.interruption = lush.Interruption:new()
		local prompt = options.prompt or "pager"
		local pager_gui = flow.make_gui(function(player, ctx)
			return gui.VBox
			{
				gui.Label{label = prompt, padding = 0.5, align_h = "center"},
				gui.HBox
				{
					gui.Label{label = string.format("type: %s", context.stdin.type), align_h = "right"}
				},
				gui.ScrollableVBox
				{
					name = "scroll",
					h = 7,
					w = 6,

					unpack(create_pager_scroll_area(context.stdin))
				},
			}
		end)

		pager_gui:show(player)
	end,
})

local function dmenu_on_event(player, ctx, value)
	ctx.context.stdout:push(value)
	ctx.context.interruption:stop()
end

local function create_dmenu_scroll_area(stdin)
	local output = {}
	local value

	for i, v in stdin:iterator() do
		value = lush.DatatypeValidator.cast_to(v, "string")
		table.insert(output, gui.ButtonExit{label = value, name = value, on_event = 
		function(player, ctx)
			dmenu_on_event(player, ctx, v)
		end})
	end

	return output
end

lush.Command:new(
{
	name = "dmenu",
	description = "displays STDIN to user to chose",
	options = {prompt = true},
	privs = {},
	arguments = {},
	stdin_type = "string",
	callback = function(context, args, options)
		local player
		if options.player and context.priv_cache["lush_display"] then
			player = lush.DatatypeValidator.cast_to(options["player"], "player")
		else
			player = minetest.get_player_by_name(context.env:get("name"))
		end

		context.interruption = lush.Interruption:new()
		local prompt = options.prompt or "DMENU"
		local dmenu_gui = flow.make_gui(function(player, ctx)
			return gui.VBox
			{
				gui.Label{label = prompt, padding = 0.5, align_h = "center"},
				gui.HBox
				{
					gui.Label{label = string.format("type: %s", context.stdin.type), align_h = "right"}
				},
				gui.ScrollableVBox
				{
					name = "scroll",
					h = 7,
					w = 6,

					unpack(create_dmenu_scroll_area(context.stdin))
				},
				gui.ButtonExit{label = "Finish", padding = 0.5, align_h = "center",
				on_event = function(player, ctx)
					context.interruption:stop()
				end},
			}
		end)

		dmenu_gui:show(player, {context = context})
	end,
})

lush.Command:new(
{
	name = "display_label",
	description = "Displays text on player's screen",
	options = {player = true},
	privs = {},
	arguments = 
	{
		{
			name = "text",
			type = "string",
			single = true
		},
		{
			name = "duration",
			type = "time",
			single = true
		},
	},
	callback = function(context, args, options)
		local player
		if options.player and context.priv_cache["lush_display"] then
			player = lush.DatatypeValidator.cast_to(options["player"], "player")
		else
			player = minetest.get_player_by_name(context.env:get("name"))
		end

		local hud_id = player:hud_add(
		{
			type = "label",
			text = args.text,
			position = {x = 0.5, y = 0.1}
		})

		minetest.after(args.duration,
		function()
			player:hud_remove(hud_id)
		end)
	end,
})

local waypoints = minetest.deserialize(storage:get_string("waypoints") or "") or {}

minetest.register_on_joinplayer(
function(player)
	local name = player:get_player_name()

	if not waypoints[name] then return end

	for waypoint_name, data in pairs(waypoints[name]) do
		waypoints[name][waypoint_name].hud_id = player:hud_add(
		{
			type = "waypoint",
			text = waypoint_name,
			world_pos = data.world_pos
		})
	end
end)

local function remove_waypoint(player_name, waypoint_name)
	local player = minetest.get_player_by_name(player_name)
	if waypoints[player_name] and waypoints[player_name][waypoint_name] then
		player:hud_remove(waypoints[player_name][waypoint_name].hud_id)
		waypoints[player_name][waypoint_name] = nil
		storage:set_string("waypoints", minetest.serialize(waypoints))
	end
end

lush.Command:new(
{
	name = "waypoint",
	description = "Adds a waypoint at pos",
	options = {player = true, expire = true},
	privs = {},
	arguments = 
	{
		{
			name = "name",
			type = "string",
			single = true
		},
		{
			name = "pos",
			type = "vector",
			single = true
		},
	},
	callback = function(context, args, options)
		local player
		if options.player and context.priv_cache["lush_display"] then
			player = lush.DatatypeValidator.cast_to(options["player"], "player")
		else
			player = minetest.get_player_by_name(context.env:get("name"))
		end

		local name = player:get_player_name()

		waypoints[name] = waypoints[name] or {}


		if waypoints[name][args.name] then
			remove_waypoint(name, args.name)
		end

		waypoints[name][args.name] =
		{
			world_pos = args.pos,
			hud_id = player:hud_add(
			{
				type = "waypoint",
				text = args.name,
				world_pos = args.pos
			})
		}

		storage:set_string("waypoints", minetest.serialize(waypoints))

		if options.expire then
			local expiration_time = lush.DatatypeValidator.cast_to(options.expire, "time")
			minetest.after(expiration_time,
			function()
				remove_waypoint(name, args.name)
			end)
		end
	end,
})

lush.Command:new(
{
	name = "remove_waypoint",
	description = "removes a waypoint",
	options = {player = true},
	privs = {},
	arguments = 
	{
		{
			name = "name",
			type = "string",
			single = true
		},
	},
	callback = function(context, args, options)
		local player
		if options.player and context.priv_cache["lush_display"] then
			player = lush.DatatypeValidator.cast_to(options["player"], "player")
		else
			player = minetest.get_player_by_name(context.env:get("name"))
		end

		local name = player:get_player_name()

		if waypoints[name] and waypoints[name][args.name] then
			remove_waypoint(name, args.name)
		else
			error("no such waypoint exists")
		end
	end,
})

lush.Command:new(
{
	name = "waypoint_pos",
	description = "gets position of a waypoint",
	options = {player = true},
	privs = {},
	arguments = 
	{
		{
			name = "name",
			type = "string",
			single = true
		},
	},
	callback = function(context, args, options)
		local player
		if options.player and context.priv_cache["lush_display"] then
			player = lush.DatatypeValidator.cast_to(options["player"], "player")
		else
			player = minetest.get_player_by_name(context.env:get("name"))
		end

		local name = player:get_player_name()

		if waypoints[name] and waypoints[name][args.name] then
			context.stdout:push(waypoints[name][args.name].world_pos)
		else
			error("no such waypoint exists")
		end
	end,
})
