-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_InlineLua
--	Describes inline lua code chunks

local InlineLua =
{
	instance_of = "LanguageConstruct_InlineLua",
}

InlineLua.__index = InlineLua

function InlineLua:new(t)
	if t == nil then
		t = {}
	end

	-- {{{ type checking
	if type(t) ~= "table" then error("t should be of type table. instead of " .. type(t)) end
	if t.chunk and type(t.chunk) ~= "function" then error("t.chunk should be of type function. instead of " .. type(t.chunk)) end
	-- }}}

	self = setmetatable(
	{
		chunk = t.chunk or {}
	}, InlineLua)

	return self
end

function InlineLua:evaluate(context)
	-- {{{ type checking
	if context.instance_of ~= "ShellContext" then error("context should be an instance of ShellContext. instead of " .. context.instance_of) end
	-- }}}

	local env_with_context = getfenv(1)
	env_with_context.context = context
	setfenv(self.chunk, env_with_context)
	return self.chunk(context)
end

function InlineLua:parse(parser_context)
	-- {{{ type checking
	if parser_context.instance_of ~= "ParserContext" then error("parser_context should be an instance of ParserContext. instead of " .. parser_context.instance_of) end
	-- }}}

	-- state: *
	parser_context:advance()
	-- state: `*

	local offset_index = 1
	local char
	local is_escaped = false
	local code_text = ""
	local error_message
	while not parser_context:is_EOF() do
		char = parser_context:consume(offset_index)

		if char == "`" and not is_escaped then
			break
		elseif char == [[\]] then
			is_escaped = true
		elseif char == "n" and is_escaped then
			code_text = code_text .. "\n"
			is_escaped = false
		else
			if is_escaped then
				code_text = code_text .. [[\]]
				is_escaped = false
			end

			code_text = code_text .. char
		end

	end

	self.chunk, error_message = loadstring(code_text, "inline lua code")

	if self.chunk == nil then
		error(error_message)
	end

	-- state: `<code>`*

end

return InlineLua
