-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

require("tests/busted_definitions")
local inspect = require("inspect")

describe("LanguageConstruct_UnquotedString class:",
function()
	local UnquotedString = require("language_constructs.unquoted_string")
	local ParameterExpansion = require("language_constructs.parameter_expansion")
	local ShellContext = require("shell_context")
	local ParserContext = require("parser_context")
	local shell_context_instance
	local parser_context_instance
	local unquoted_string_instance
	local terminator_set =
	{
		[" "] = true
	}

	before_each(
	function()
		shell_context_instance = ShellContext:new()
		parser_context_instance = ParserContext:new()
		unquoted_string_instance = UnquotedString:new()
	end)

	describe("evaluates correctly: ",
	function()

		it("when given a simple string",
		function()
			unquoted_string_instance.elements = {"string"}

			assert.equals("string", unquoted_string_instance:evaluate(shell_context_instance))
		end)

		it("when given a string with an expansion",
		function()
			unquoted_string_instance.elements = {"string", ParameterExpansion:new({parameter_name = "parameter"})}

			shell_context_instance.env["parameter"] = "value"

			assert.equals("stringvalue", unquoted_string_instance:evaluate(shell_context_instance))
		end)
	end)

	describe("parses correctly: ",
	function()

		it("when given a simple string",
		function()
			parser_context_instance.text = [[string]]

			unquoted_string_instance:parse(parser_context_instance, terminator_set)

			assert.same({"string"}, unquoted_string_instance.elements)
		end)

		it("when given a string with an expansion",
		function()
			parser_context_instance.text = [[string${parameter}string]]

			unquoted_string_instance:parse(parser_context_instance, terminator_set)

			assert.equals("string", unquoted_string_instance.elements[1])

			assert.equals("parameter", unquoted_string_instance.elements[2].parameter_name)
			assert.equals(0, unquoted_string_instance.elements[2].modifier)
			assert.same({}, unquoted_string_instance.elements[2].modifier_args)

			assert.equals("string", unquoted_string_instance.elements[3])

			assert.equals(3, #unquoted_string_instance.elements)
		end)

		it("stops parsing after a terminator",
		function()
			parser_context_instance.text = [[string another_string]]

			unquoted_string_instance:parse(parser_context_instance, terminator_set)

			assert.same({"string"}, unquoted_string_instance.elements)

			assert.equals(7, parser_context_instance.character_index)

			assert.equals(1, #unquoted_string_instance.elements)
		end)

		it("stops parsing after a terminator when the last part was an expansion",
		function()
			parser_context_instance.text = [[string${parameter} another_string]]

			unquoted_string_instance:parse(parser_context_instance, terminator_set)

			assert.equals("string", unquoted_string_instance.elements[1])

			assert.equals("parameter", unquoted_string_instance.elements[2].parameter_name)
			assert.equals(0, unquoted_string_instance.elements[2].modifier)
			assert.same({}, unquoted_string_instance.elements[2].modifier_args)

			assert.equals(19, parser_context_instance.character_index)

			assert.equals(2, #unquoted_string_instance.elements)
		end)

		it("respects escape sequences on $ character",
		function()
			parser_context_instance.text = [[string\${parameter} another_string]]

			unquoted_string_instance:parse(parser_context_instance, terminator_set)

			assert.same({"string${parameter}"}, unquoted_string_instance.elements)

			assert.equals(20, parser_context_instance.character_index)

			assert.equals(1, #unquoted_string_instance.elements)
		end)

		it("respects escape sequences on characters in the terminator set",
		function()
			parser_context_instance.text = [[string\ another_string]]

			unquoted_string_instance:parse(parser_context_instance, terminator_set)

			assert.equals("string another_string", unquoted_string_instance.elements[1])

			assert.equals(23, parser_context_instance.character_index)

			assert.equals(1, #unquoted_string_instance.elements)
		end)

		it("respects escape sequences on basckslashes",
		function()
			parser_context_instance.text = [[string\\ another_string]]

			unquoted_string_instance:parse(parser_context_instance, terminator_set)

			assert.equals([[string\]], unquoted_string_instance.elements[1])

			assert.equals(9, parser_context_instance.character_index)

			assert.equals(1, #unquoted_string_instance.elements)
		end)

	end)

	describe("members are correctly set by constructor: ",
	function()
		it("elements",
		function()
			unquoted_string_instance = UnquotedString:new({elements = {"string"}})

			assert.same({"string"}, unquoted_string_instance.elements)
		end)

		it("instance_of",
		function()
			unquoted_string_instance = UnquotedString:new()

			assert.equals("LanguageConstruct_UnquotedString", unquoted_string_instance.instance_of)
		end)
	end)

end)
