-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local lush_override_commands = minetest.settings:get_bool("lush_override_commands")

-- setting default value
if lush_override_commands == nil then
	lush_override_commands = true
end

if lush_override_commands then
	minetest.unregister_chatcommand("mods")
end

minetest.register_privilege("lush_heal",
{
	description = "whether the player can use the heal command",
	give_to_singleplayer = false
})

minetest.register_privilege("lush_damage",
{
	description = "whether the player can use the damage/kill command",
	give_to_singleplayer = false
})

minetest.register_privilege("invisible",
{
	description = "whether the player can use the invis command",
	give_to_singleplayer = false
})

minetest.register_privilege("clear",
{
	description = "whether the player can use the clear command",
	give_to_singleplayer = false
})

minetest.register_privilege("fix",
{
	description = "whether the player can use the fix_items command",
	give_to_singleplayer = false
})

local max = tonumber(minetest.settings:get("lush_max_lag")) or 2.0

local lag = 0
local last = minetest.get_us_time()
minetest.register_globalstep(function()
		if lag <= 0 then 
			return
		end

		local exp = last + lag * 1000000
		while minetest.get_us_time() < exp do 
			-- looping until time passes
		end

		last = exp
	end)

lush.Command:new(
{
	name = "lag",
	description = "lags out the server so each global step takes a specific amount of seconds",
	privs = {server = true},
	arguments = 
	{
		{
			name = "time_per_globalstep",
			type = "time",
			single = true
		}
	},
	callback = function(context, args, options)
		if args.time_per_globalstep > max then
			error("time per globalstep can not exceed " .. tostring(max) .. " seconds")
		end

		lag = args.time_per_globalstep
	end,
})

lush.Command:new(
{
	name = "days",
	description = "prints the amounts of days passed",
	privs = {debug = true},
	arguments = {},
	callback = function(context, args, options)
		context.stdout:push(minetest.get_day_count())
	end,
})

lush.Command:new(
{
	name = "set_node",
	description = "sets a node",
	privs = {give = true},
	arguments = 
	{
		{
			name = "pos",
			type = "vector",
			single = false
		},
		{
			name = "node",
			type = "node_name",
			single = true
		},
	},
	callback = function(context, args, options)
		for i, pos in args.pos:iterator() do
			minetest.set_node(pos, {name = args.node})
		end
	end,
})

local storage = minetest.get_mod_storage()

local time_update_job

local function time_update_func(time)
	minetest.set_timeofday(time)
	time_update_job = minetest.after(1, time_update_func, time)
end

if storage:contains("lush_permament_time") and storage:get_float("lush_permament_time") ~= -1 then
	time_update_job = minetest.after(1, time_update_func, storage:get_float("lush_permament_time"))
end

lush.Command:new(
{
	name = "time",
	description = "sets time",
	options = {["permament"] = false},
	privs = {settime = true},
	arguments =
	{
		{
			name = "time",
			type = {"time", "string"},
			single = true
		}
	},
	callback = function(context, args, options)
		local time_of_day = 0

		if type(args.time) == "string" then
			if args.time == "midnight" then
				time_of_day = 0
			elseif args.time == "night" then
				time_of_day = 0.8
			elseif args.time == "sunset" then
				time_of_day = 0.75
			elseif args.time == "noon" then
				time_of_day = 0.5
			elseif args.time == "day" then
				time_of_day = 0.3
			elseif args.time == "sunrise" then
				time_of_day = 0.2
			else
				error(string.format("%s isnt a recgonized time", args.time))
			end
		else
			if args.time >= 86400 or args.time < 0 then
				error("time must be between 0 seconds and 24 hours")
			end

			time_of_day = args.time / 86400 -- seconds in a day
		end

		minetest.set_timeofday(time_of_day)

		if options["permament"] then
			if time_update_job then
				time_update_job:cancel()
			end
			time_update_job = minetest.after(1, time_update_func, time_of_day)
			storage:set_float("lush_permament_time", time_of_day)
		else
			if time_update_job then
				time_update_job:cancel()
			end
			storage:set_float("lush_permament_time", -1)
		end
	end,
})

lush.Command:new(
{
	name = "launch",
	description = "adds velocity to entities",
	privs = {lush_damage = true},
	arguments =
	{
		{
			name = "objects",
			type = "objectref",
			single = false
		},
		{
			name = "vector",
			type = "vector",
			single = true
		}
	},
	callback = function(context, args, options)
		for i, v in args.objects:iterator() do
			v:add_velocity(args.vector)
		end
	end,
})

lush.Command:new(
{
	name = "teleport",
	description = "teleports entities to position",
	options = {},
	privs = {teleport = true},
	arguments =
	{
		{
			name = "objects",
			type = "objectref",
		},
		{
			rule_type = "terminal",
			value = "to",
			optional = true,
		},
		{
			name = "pos",
			type = "vector",
			single = true
		}
	},
	callback = function(context, args, options)
		local lm = 31007 -- equals MAX_MAP_GENERATION_LIMIT in C++
		local pos = args.pos
		if pos.x < -lm or pos.x > lm or pos.y < -lm or pos.y > lm
				or pos.z < -lm or pos.z > lm then
			error("position out of map's bounds")
		end

		if (context.env:get("name") ~= args.objects:get(1):get_player_name() or #args.objects > 1) and not minetest.check_player_privs(context.env:get("name"), "bring") then
			error("you need the 'bring' privilege to teleport anything other than yourself")
		end

		for i, v in args.objects:iterator() do
			if v:get_pos() and not v:get_attach() then
				v:set_pos(args.pos)
			end
		end

	end,
})

lush.Command:new(
{
	name = "kill",
	description = "kills entities",
	privs = {lush_damage = true},
	arguments =
	{
		{
			name = "objects",
			type = "objectref",
		},
	},
	callback = function(context, args, options)
		for i, v in args.objects:iterator() do
			lush.helpers.deal_damage(v, v:get_hp(), nil, true)
		end

	end,
})

lush.Command:new(
{
	name = "damage",
	description = "deals damage to entities",
	options =
	{
		["type"] = true,
	},
	privs = {lush_damage = true},
	arguments =
	{
		{
			name = "targets",
			type = "objectref",
			single = false,
		},
		{
			name = "amount",
			type = "number",
			single = true,
		},
	},
	callback = function(context, args, options)
		local damage_type = "set_hp" or options["type"]

		if type(damage_type) ~= "string" then
			error("damage type should be a string")
		end

		for i, v in args.targets:iterator() do
			lush.helpers.deal_damage(v, args.amount, nil, true)
		end

	end,
})

lush.Command:new(
{
	name = "spawnpoint",
	description = "sets the respawn point for players",
	privs = {server = true},
	arguments =
	{
		{
			name = "names",
			type = "player_name",
			single = false,
		},
		{
			name = "pos",
			type = "vector",
			single = true,
		},
	},
	callback = function(context, args, options)
		for _, v in args.names:iterator() do
			lush.spawnpoints[v] = args.pos
		end

		lush.Storage.save("spawnpoints", lush.spawnpoints)
	end,
})

lush.Command:new(
{
	name = "clear_spawnpoint",
	description = "removes the previously set spawnpoint",
	privs = {server = true},
	arguments =
	{
		{
			name = "names",
			type = "player_names",
			single = false,
		},
	},
	callback = function(context, args, options)
		for _, v in args.names:iterator() do
			lush.spawnpoints[v] = nil
		end

		lush.Storage.save("spawnpoints", lush.spawnpoints)
	end,
})

lush.Command:new(
{
	name = "heal",
	description = "heals entities by an amount",
	privs = {lush_heal = true},
	arguments =
	{
		{
			name = "targets",
			type = "objectref",
			single = false,
		},
		{
			name = "amount",
			type = "number",
			single = true,
		},
	},
	callback = function(context, args, options)
		for i, v in args.targets:iterator() do
			lush.helpers.deal_damage(v, -args.amount, nil, false)
		end
	end,
})

lush.Command:new(
{
	name = "list",
	description = "lists online players",
	privs = {debug = true},
	arguments = {},
	callback = function(context, args, options)
		for _, player in ipairs(minetest.get_connected_players()) do
			context.stdout:push(player:get_player_name())
		end
	end,
})

lush.Command:new(
{
	name = "seed",
	description = "prints the world's seed",
	privs = {debug = true},
	arguments = {},
	callback = function(context, args, options)
		context.stdout:push(minetest.get_mapgen_setting("seed"))
	end,
})

lush.Command:new(
{
	name = "lsmods",
	description = "prints active mods to STDOUT",
	privs = {debug = true},
	arguments = {},
	callback = function(context, args, options)
		for i, v in ipairs(minetest.get_modnames()) do
			context.stdout:push(v)
		end
	end,
})

-- credit to SkyBuilder1717's Essentials mod
-- from whom this implementation is partly ripped
lush.Command:new(
{
	name = "invis",
	description = "makes the entities invisible",
	privs = {invisible = true},
	arguments =
	{
		{
			type = "player",
			name = "players",
			single = false
		},
		{
			type = "string",
			name = "is_enabled",
			single = true
		}
	},
	callback = function(context, args, options)
		local prop

		if args.is_enabled == "on" then
			prop = {
				visual_size = {x = 0, y = 0, z = 0},
				is_visible = false,
				nametag_color = {r=255,g=255,b=255,a=255},
				pointable = false,
				makes_footstep_sound = false,
				show_on_minimap = false,
			}
		elseif args.is_enabled == "off" then
			prop = {
				visual_size = {x = 1, y = 1, z = 1},
				is_visible = true,
				nametag_color = {r=255,g=255,b=255,a=0},
				pointable = true,
				makes_footstep_sound = true,
				show_on_minimap = true,
			}
		else
			error("invalid argument: '" .. tostring(args.is_enabled) .. "', should be either on or off")
		end

		for i, v in args.players:iterator() do
			v:set_properties(prop)

			if args.is_enabled == "on" then
				v:get_meta():set_int("invisible", 1)
			elseif args.is_enabled == "off" then
				v:get_meta():set_int("invisible", 0)
			end
		end
	end,
})

lush.Command:new(
{
	name = "give",
	description = "gives items",
	options =
	{
		["wear"] = true,
		["amount"] = true,
		["meta"] = true
	},
	privs = {give = true},
	arguments =
	{
		{
			type = "player",
			name = "players",
			single = false
		},
		{
			type = {"node_name", "string"},
			name = "item_name",
			single = true
		},
	},
	callback = function(context, args, options)
		local stack = ItemStack({name = args.item_name})

		if options.meta then
			stack:get_meta():from_table({fields = options.meta})
		end

		if options.wear then
			stack:set_wear(options.wear)
		end

		if options.amount then
			stack:set_count(options.amount)
		end

		local inv
		for i, object in args.players:iterator() do
			inv = object:get_inventory()

			inv:add_item("main", stack)
		end

	end,
})

lush.Command:new(
{
	name = "summon",
	description = "summons entities",
	options =
	{
		["count"] = true,
	},
	privs = {give = true},
	arguments =
	{
		{
			type = "string",
			name = "entity_name",
			single = true
		},
		{
			type = "vector",
			name = "pos",
			single = true
		},
	},
	callback = function(context, args, options)
		local success
		if options.count then
			for i = 1, options.count do
				success = minetest.add_entity(args.pos, args.entity_name)
			end
		else
			success = minetest.add_entity(args.pos, args.entity_name)
		end

		if not success then
			error("couldnt summon entity")
		end
	end,
})

lush.Command:new(
{
	name = "pulverize",
	description = "clears items",
	privs = {clear = true},
	arguments =
	{
		{
			name = "items",
			type = "inventory_item_list",
			single = true
		},
	},
	callback = function(context, args, options)
		local lists = args.items:get_lists()

		for listname, stacks in pairs(lists) do
			for _, stack in pairs(stacks) do
				stack:clear()
			end
		end

		args.items:set_lists(lists)
	end,
})

lush.Command:new(
{
	name = "fix_items",
	description = "fixes items",
	privs = {fix = true},
	arguments =
	{
		{
			name = "items",
			type = "inventory_item_list",
			single = true
		},
	},
	callback = function(context, args, options)
		local lists = args.items:get_lists()

		for listname, stacks in pairs(lists) do
			for _, stack in pairs(stacks) do
				stack:set_wear(0)
			end
		end

		args.items:set_lists(lists)
	end,
})

lush.Command:new(
{
	name = "itemmeta",
	description = "prints item metadata",
	privs = {debug = true},
	arguments =
	{
		{
			name = "items",
			type = "inventory_item_list",
			single = true
		},
	},
	callback = function(context, args, options)
		local lists = args.items:get_lists()

		for listname, stacks in pairs(lists) do
			for _, stack in pairs(stacks) do
				context.stdout:push(stack:get_meta():to_table())
			end
		end
	end,
})

lush.Command:new(
{
	name = "setitemmeta",
	description = "sets item metadata",
	privs = {give = true},
	arguments =
	{
		{
			name = "items",
			type = "inventory_item_list",
			single = true
		},
		{
			name = "meta",
			type = "table",
			single = true
		},
	},
	callback = function(context, args, options)
		local lists = args.items:get_lists()

		for listname, stacks in pairs(lists) do
			for _, stack in pairs(stacks) do
				stack:get_meta():from_table({fields = args.meta})
			end
		end

		args.items:set_lists(lists)
	end,
})

lush.Command:new(
{
	name = "lsitems",
	description = "lists items",
	options = {["inventory-format"] = false},
	arguments =
	{
		{
			name = "items",
			type = "inventory_item_list",
			single = true
		},
	},
	callback = function(context, args, options)

		if options["inventory-format"] then
				for list_name, list in pairs(args.items:get_lists()) do
					context.stdout:push("List " .. list_name)
					for _, v in pairs(list) do
						context.stdout:push(v:to_string())
					end
					context.stdout:push("EndInventoryList")
				end
		else
			-- print in tree like format
			-- the code looks funny because the last element must start with
			-- an "└─ " even if the last item in the inventory is empty
			-- this is a very clever solution to this problem
			for list_name, list in pairs(args.items:get_lists()) do
				context.stdout:push(list_name)

				local previous_item
				for i, item in pairs(list) do

					if not item:is_empty() then
						if previous_item then
							context.stdout:push("├─ " .. previous_item:to_string())
						end
						previous_item = item
					end
				end

				if previous_item then
					context.stdout:push("└─ " .. previous_item:to_string())
				end
			end
		end
	end,
})

lush.Command:new(
{
	name = "basename",
	description = "prints the technical name of a node",
	arguments =
	{
		{
			name = "node_name",
			type = "node_name",
			single = true
		},
	},
	callback = function(context, args, options)
			context.stdout:push(args.node_name)
	end,
})

lush.Command:new(
{
	name = "dump_luaentity",
	description = "prints the lua entity of a given object",
	privs = {debug = true},
	arguments =
	{
		{
			name = "object",
			type = "objectref",
			single = true
		},
	},
	callback = function(context, args, options)
		if args.object:get_luaentity() then
			context.stdout:push(args.object:get_luaentity())
		else
			error("couldnt get lua entity")
		end
	end,
})

lush.Command:new(
{
	name = "dump",
	description = "dumps STDIN into the debug file for inspection",
	privs = {debug = true},
	arguments = {},
	stdin_type = "string",
	callback = function(context, args, options)
		minetest.log("action", string.format("%s dumped information:\n%s", 
			context.env:get("name"), table.concat(context.stdin.container, "\n")))
	end,
})

lush.Command:new(
{
	name = "nodeinfo",
	description = "prints information about the node at a given position",
	options =
	{
		["human"] = false
	},
	privs = {debug = true},
	arguments =
	{
		{
			rule_type = "terminal",
			name = "type",
			value =
			{
				["metadata"] = true,
				["inventory"] = true,
				["groups"] = true,
				["name"] = true,
				["param1"] = true,
				["param2"] = true,
			}
		},
		{
			name = "pos",
			type = "vector",
			single = true
		}
	},
	callback = function(context, args, options)

		if args.type == "metadata" then
			context.stdout:push(minetest.get_meta(args.pos):to_table())
		elseif args.type == "inventory" then
			local inv = minetest.get_inventory({type = "node", pos = args.pos})
			local indexes = {}

			for list_name, list in pairs(inv:get_lists()) do
				indexes[list_name] = {}
				for i, _ in pairs(list) do
					table.insert(indexes[list_name], i)
				end
			end

			context.stdout:push(lush.InventoryItemList:new(
			{
				location = {type = "node", pos = args.pos},
				indexes = indexes
			}))
		elseif args.type == "groups" then
			local nodedef = minetest.registered_nodes[minetest.get_node(args.pos).name]

			for k, v in pairs(nodedef.groups) do
				context.stdout:push(tostring(k) .. "=" .. tostring(v))
			end
		elseif args.type == "name" then
			context.stdout:push(minetest.get_node(args.pos).name)
		elseif args.type == "param1" then
			context.stdout:push(minetest.get_node(args.pos).param1)
		elseif args.type == "param2" then
			context.stdout:push(minetest.get_node(args.pos).param2)
		end
	end,
})

lush.Command:new(
{
	name = "setnodeinfo",
	description = "sets properties of nodes",
	options =
	{
		["human"] = false
	},
	privs = {debug = true, give = true},
	arguments =
	{
		{
			rule_type = "terminal",
			name = "type",
			value =
			{
				["metadata"] = true,
				["inventory"] = true,
				["groups"] = true,
				["name"] = true,
				["param1"] = true,
				["param2"] = true,
			}
		},
		{
			name = "pos",
			type = "vector",
			single = true
		},
		{
			name = "value",
			single = true
		}
	},
	callback = function(context, args, options)
		local success

		if args.type == "metadata" then
			args.value, success = lush.DatatypeValidator.cast_to(args.value, "table")

			if not success then
				error("invalid argument #3: invalid type")
			end

			minetest.get_meta(args.pos):from_table(args.value)
		elseif args.type == "inventory" then
			args.value, success = lush.DatatypeValidator.cast_to(args.value, "inventory_item_list")

			if not success then
				error("invalid argument #3: invalid type")
			end

			local inv = minetest.get_inventory({type = "node", pos = args.pos})
			local inv_list = inv:get_lists()

			for listname, list in pairs(args.value:get_lists()) do
				for index, stack in pairs(list) do
					inv_list[listname][index] = stack
				end
			end

			inv:set_lists(inv_list)
		elseif args.type == "name" then
			args.value, success = lush.DatatypeValidator.cast_to(args.value, "string")

			if not success then
				error("invalid argument #3: invalid type")
			end

			local node = minetest.get_node(args.pos)
			minetest.set_node(args.pos, {name = args.value, param2 = node.param2, param1 = node.param1})
		elseif args.type == "param1" then
			args.value, success = lush.DatatypeValidator.cast_to(args.value, "number")

			if not success then
				error("invalid argument #3: invalid type")
			end

			local node = minetest.get_node(args.pos)
			minetest.set_node(args.pos, {name = node.name, param2 = node.param2, param1 = args.value})
		elseif args.type == "param2" then
			args.value, success = lush.DatatypeValidator.cast_to(args.value, "number")

			if not success then
				error("invalid argument #3: invalid type")
			end

			local node = minetest.get_node(args.pos)
			minetest.set_node(args.pos, {name = node.name, param2 = args.value, param1 = node.param1})
		end
	end,
})
