-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

local helpers = require("helpers")
local Expression = require("language_constructs.expression")
local Shell = require("language_constructs.shell")
local DatatypeValidator = require("datatype_validator")

local WhileLoop =
{
	instance_of = "LanguageConstruct_WhileLoop",
}

WhileLoop.__index = WhileLoop

function WhileLoop:new(t)
	t = t or {}
	self = setmetatable(
	{
		expression = t.expression or nil,
		subshell = t.subshell or nil
	}, WhileLoop)

	return self
end

local function evaluate_as_bool(value)
	if type(value) == "table" and value.instance_of == "TypedList" then
		return #value.container ~= 0
	end

	local datatype = DatatypeValidator.get_type_of(value)

	if type(value) == "table" and value.instance_of == "TypedList" then
		return #value > 0
	elseif DatatypeValidator.registered_types[datatype]
		and DatatypeValidator.registered_types[datatype].to_bool then
		return DatatypeValidator.registered_types[datatype].to_bool(value)
	else
		return false
	end
end

function WhileLoop:evaluate(context, max_frustration)
	-- {{{ type checking
	if context.instance_of ~= "ShellContext" then error("context should be an instance of ShellContext. instead of " .. context.instance_of) end
	-- }}}

	max_frustration = max_frustration or 10000
	local frustration = 0

	while evaluate_as_bool(self.expression:evaluate(context)) do
		if frustration > max_frustration then
			error("loop exceeded maximum iterations, as defined by lush_max_loop_iterations setting")
		end

		self.subshell:evaluate(context, true)

		frustration = frustration + 1
	end
end

function WhileLoop:parse(parser_context)
	-- {{{ type checking
	if parser_context.instance_of ~= "ParserContext" then error("parser_context should be an instance of ParserContext. instead of " .. parser_context.instance_of) end
	-- }}}

	local success, match = parser_context:match_fast("while", true)

	if not success then
		error("expected 'while'")
	end

	parser_context:skip_until(helpers.white_chars_set, true)

	self.expression = Expression:new()
	self.expression:parse(parser_context, helpers.white_chars_set)

	parser_context:skip_until(helpers.white_chars_set, true)

	local success, match = parser_context:match_fast("do", true)

	if not success then
		error("expected 'do'")
	end

	parser_context:skip_until(helpers.white_chars_set, true)

	self.subshell = Shell:new()
	self.subshell:parse(parser_context, {"end"})

	local success, match = parser_context:match_fast("end", true)

	if not success then
		error("bruh?")
	end
end

return WhileLoop
