-- Copyright (C) 2024 rstcxk
-- 
-- This program is free software: you can redistribute it and/or modify it under the terms of
-- the GNU Affero General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
-- without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more details.
-- 
-- You should have received a copy of the GNU Affero General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 

--- @classmod LanguageConstruct_CommandInvocation
--	Describes a command invocation

local inspect = require("inspect")
local helpers = require("helpers")

local StringExpression
local Expression

local CommandInvocation =
{
	instance_of = "LanguageConstruct_CommandInvocation",
}

CommandInvocation.__index = CommandInvocation

function CommandInvocation:new(t)
	if t == nil then
		t = {}
	end

	-- {{{ type checking
	if type(t) ~= "table" then error("t should be of type table. instead of " .. type(t)) end
	if t.args and type(t.args) ~= "table" then error("t.args should be of type table. instead of " .. type(t.args)) end
	-- }}}

	self = setmetatable(
	{
		name = t.name or nil, --i dont want to require the StringExpression module here, because i doubt it would ever be useful
		args = t.args or {}
	}, CommandInvocation)

	return self
end

function CommandInvocation:evaluate(context)
	-- {{{ type checking
	if context.instance_of ~= "ShellContext" then error("context should be an instance of ShellContext. instead of " .. context.instance_of) end
	-- }}}

	-- i know, it gets confusing when i have two objects with the same name, but idk how else to name it
	local Command= require("command")

	local command_name = self.name:evaluate(context)
	local command = Command.get_command_by_name(command_name)

	local args = {}

	if command then
		for i, argument in ipairs(self.args) do
			table.insert(args, argument:evaluate(context))
		end

		command:call(context, args)
	else
		error("command not found: " .. command_name)
	end
end

function CommandInvocation:parse(parser_context, terminator_set)
	-- {{{ type checking
	if parser_context.instance_of ~= "ParserContext" then error("parser_context should be an instance of ParserContext. instead of " .. parser_context.instance_of) end
	-- }}}

	StringExpression = StringExpression or require("language_constructs.string_expression")
	Expression = Expression or require("language_constructs.expression")

	local arg
	local matching, _

	-- print("COMMAND TERMINATORS", inspect(terminator_set))

	self.name = StringExpression:new()
	self.name:parse(parser_context, helpers.concat_sets(helpers.white_chars_set, terminator_set))

	-- state: <command_name>*

	-- local frustration = 5
	-- print(dump(Expression))
	-- print(dump(Expression:new()))

	while true do
		-- frustration = frustration - 1
		-- if frustration < 0 then error("SHIT") end
		parser_context:skip_until(helpers.white_chars_set, true)
		matching, _ = parser_context:match(helpers.control_operators_list)

		-- state: <command_name><arguments if any><whitespace>*

		if matching or parser_context:is_EOF() then
			-- state: <command_name><arguments>*<control operator>
			break
		elseif terminator_set[parser_context:peek()] then
			break
		end

		-- print(parser_context:peek())
		-- print(string.sub(parser_context.text, parser_context.character_index, #parser_context.text))
		-- print(parser_context.text)
		-- print(parser_context.character_index)
		-- print(inspect(self))


		arg = Expression:new()
		arg:parse(parser_context, helpers.concat_sets(helpers.white_chars_set, terminator_set))
		table.insert(self.args, arg)

		-- state: <command_name><arguments if any><whitespace><argument>*
	end
end

return CommandInvocation
